#
#define true 0177777
#define false 000000
#define then /* */

#include "../param.h"
#include "../systm.h"
#include "../filsys.h"
#include "../inode.h"
#include "../user.h"
#include "../../shared/constants.h"

/*
 * iinit is called once (from main)
 * very early in initialization.
 * It reads the root's super block
 * and initializes the current date
 * from the last modified date.
 *
 * panic: iinit -- cannot read the super
 * block. Usually because of an IO error.
 */
iinit()
{
	register *cp, *bp;

	blkopen( rootdev , _READ_ACCESS );
	bp = bfetch(rootdev, 1, _READ_ACCESS);
	if(u.u_error)
		panic("iinit");
	cp = &superblocks[0];
	bcopy(bp, cp, 256);
	brelse(bp);
	mount[0].m_inuse = true;
	mount[0].m_dev = rootdev;
	cp->s_flock = false;
	cp->s_ilock = false;
	cp->s_ronly = true;
	time[0] = cp->s_time[0];
	time[1] = cp->s_time[1];
}

/*
 * alloc will obtain the next available
 * free disk block from the free list of
 * the specified device.
 * The super block has up to 100 remembered
 * free blocks; the last of these is read to
 * obtain 100 more . . .
 *
 * no space on dev x/y -- when
 * the free list is exhausted.
 */
alloc(dev)
{
	int bno;
	register *bp, *ip, *fp;

	fp = getfs(dev);
	if (fp->s_ronly)
		return( NULL );
	while(fp->s_flock)
		idle();
	do {
		if(fp->s_nfree <= 0)
			goto nospace;
		bno = fp->s_free[--fp->s_nfree];
		if(bno == 0)
			goto nospace;
	} while (badblock(fp, bno, dev));
	if(fp->s_nfree <= 0) {
		fp->s_flock++;
		bp = bfetch(dev, bno, _READ_WRITE_ACCESS);
		if (u.u_error)
			return( NULL );
		ip = bp;
		fp->s_nfree = *ip++;
		bcopy(ip, fp->s_free, 100);
		brelse(bp);
		fp->s_flock = false;
	}
	bclear( dev, bno );
	return( bno );

nospace:
	fp->s_nfree = 0;
	prdev("no space", dev);
	u.u_error = ENOSPC;
	return(NULL);
}

/*
 * place the specified disk block
 * back on the free list of the
 * specified device.
 */
free(dev, bno)
{
	register *fp, *bp, *ip;

	fp = getfs(dev);
	if (fp->s_ronly)
		return( NULL );
	while(fp->s_flock)
		idle();
	if (badblock(fp, bno, dev))
		return;
	if(fp->s_nfree <= 0) {
		fp->s_nfree = 1;
		fp->s_free[0] = 0;
	}
	if(fp->s_nfree >= 100) {
		fp->s_flock++;
		bp = bfetch(dev, bno, _READ_WRITE_ACCESS);
		if (u.u_error)
			return;
		ip = bp;
		*ip++ = fp->s_nfree;
		bcopy(fp->s_free, ip, 100);
		fp->s_nfree = 0;
		brelse(bp);
		fp->s_flock = false;
	}
	fp->s_free[fp->s_nfree++] = bno;
	fp->s_fmod = true;
}

/*
 * Check that a block number is in the
 * range between the I list and the size
 * of the device.
 * This is used mainly to check that a
 * garbage file system has not been mounted.
 *
 * bad block on dev x/y -- not in range
 */
badblock(afp, abn, dev)
{
	register struct filsys *fp;
	register char *bn;

	fp = afp;
	bn = abn;
	if (bn < fp->s_isize+2 || bn >= fp->s_fsize) {
		prdev("bad block", dev);
		return(1);
	}
	return(0);
}

/*
 * Allocate an unused I node
 * on the specified device.
 * Used with file creation.
 * The algorithm keeps up to
 * 100 spare I nodes in the
 * super block. When this runs out,
 * a linear search through the
 * I list is instituted to pick
 * up 100 more.
 */
ialloc(dev)
{
	register *fp, *bp, *ip;
	int i, j, k, ino;

	fp = getfs(dev);
	if (fp->s_ronly)
		return( NULL );
	while(fp->s_ilock)
		idle();
loop:
	if(fp->s_ninode > 0) {
		ino = fp->s_inode[--fp->s_ninode];
		ip = iget(dev, ino);
		if (ip==NULL)
			return(NULL);
		if(ip->i_mode == 0) {
			for(bp = &ip->i_mode; bp < &ip->i_addr[8];)
				*bp++ = 0;
			fp->s_fmod = true;
			return(ip);
		}
		/*
		 * Inode was allocated after all.
		 * Look some more.
		 */
		iput(ip);
		goto loop;
	}
	fp->s_ilock++;
	ino = 0;
	for(i=0; i<fp->s_isize; i++) {
		bp = bfetch(dev, i+2, _READ_WRITE_ACCESS);
		if (u.u_error)
			return(NULL);
		ip = bp;
		for(j=0; j<256; j=+16) {
			ino++;
			if(ip[j] != 0)
				continue;
			for(k=0; k<NINODE; k++)
			if(dev==inode[k].i_dev && ino==inode[k].i_number)
				goto cont;
			fp->s_inode[fp->s_ninode++] = ino;
			if(fp->s_ninode >= 100)
				break;
		cont:;
		}
		brelse(bp);
		if(fp->s_ninode >= 100)
			break;
	}
	fp->s_ilock = false;
	if (fp->s_ninode > 0)
		goto loop;
	prdev("Out of inodes", dev);
	u.u_error = ENOSPC;
	return(NULL);
}

/*
 * Free the specified I node
 * on the specified device.
 * The algorithm stores up
 * to 100 I nodes in the super
 * block and throws away any more.
 */
ifree(dev, ino)
{
	register *fp;

	fp = getfs(dev);
	if (fp->s_ronly)
		return;
	if(fp->s_ilock)
		return;
	if(fp->s_ninode >= 100)
		return;
	fp->s_inode[fp->s_ninode++] = ino;
	fp->s_fmod = true;
}

/*
 * getfs maps a device number into
 * a pointer to the incore super
 * block.
 * The algorithm is a linear
 * search through the mount table.
 * A consistency check of the
 * in core free-block and i-node
 * counts.
 *
 * bad count on dev x/y -- the count
 *	check failed. At this point, all
 *	the counts are zeroed which will
 *	almost certainly lead to "no space"
 *	diagnostic
 * panic: no fs -- the device is not mounted.
 *	this "cannot happen"
 */
getfs(dev)
{
	register int i;
	register char *n1, *n2;

	for( i = 0; i < NMOUNT; i++ )
	if (mount[i].m_inuse && (mount[i].m_dev == dev))
	{	i = &superblocks[i];
		n1 = i->s_nfree;
		n2 = i->s_ninode;
		if(n1 > 100 || n2 > 100) {
			prdev("bad count", dev);
			i->s_nfree = 0;
			i->s_ninode = 0;
		}
		return(i);
	}
	panic("no fs");
}

/*
 * update is the internal name of the sync call
 *	It goes through the I nodes to write out any that need to be
 *	updated on disk, and updates any superblocks that have been
 *	modified
 */
update()
{
	register *p;
	register int i;
	register *bp;

	for(i = 0; i < NMOUNT; i++)
		if (mount[i].m_inuse)
		{	p = &superblocks[i];
			if ( p->s_fmod == 0 || p->s_ilock ||
			     p->s_flock     || p->s_ronly )
			then	continue;
			/* this superblock needs updating */
			bp = bfetch( mount[i].m_dev, 1, _READ_WRITE_ACCESS );
			if (u.u_error)
			then	panic("update");
			p->s_fmod = 0;
			p->s_time[0] = time[0];
			p->s_time[1] = time[1];
			bcopy( p , bp , 256 );
			brelse( bp );
		}

	for( i = 0; i < NINODE; i++ )
	{	p = &inode[i];
		if (p->i_flag&ILOCK) continue;
		p->i_flag =| ILOCK;
		iupdat( p , time );
		prele( p );
	}
}
