#
/*****************************************************************
 **								**
 **		   U C L A  Data Secure Unix			**
 **								**
 **			Copyright 1977				**
 **								**
 **    Mark Kampe, Charles Kline, Gerald Popek, Evelyn Walton	**
 **								**
 *****************************************************************/
#define true 0177777
#define false 000000
#define then  /* */
#include "../../shared/constants.h"
#include "../../shared/shared.h"
#include "../param.h"
#include "../user.h"
#include "../blkdev.h"
#include "../caps.h"
#include "../seg.h"
/* name:
	bfetch

function:
	to read a block of disk (short segment) into my address space

algorithm:
	Ascertain that I have access to the specified device;
	get a capability for the segment;
	get a page frame to put it in;
	map the segment into the page frame;

parameters:
	major and minor device number
	physical page number
	access

returns:
	address of the page in core (Super D of course)
	and a setting of u.u_error.  If nonzero, returned address is zero

globals:
	blkdevtab	list of available block devices

calls:
	getcap		to get a capability for the desired segment
	getpageframe	to get an empty frame in my virtual space
	setframe	to fill in the page frame information
	k_map		to map the segment into my virtual space

called by:
	everyone

history:
	Designed and coded by mark Kampe 11/76
*/
bfetch( dev , bn , access )
 int dev;
 int bn;
 int access;
{	register int frame;
	register int capx;

	if (blkdevtab[dev].b_open == false)
	{	u.u_error = ENODEV;
		return( NULL );
	}

	if ((blkdevtab[dev].b_access&access) != access ) then
	{	u.u_error = EACCES;
		return( NULL );
	}

	capx = getcap( dev , bn , _SMALL_SEGMENT , access );
	if (capx < 0) then
	{	u.u_error = ENOCAP;
		return( NULL );
	}

	frame = getpageframe( SUPERD );
	if (frame < 0)
	then	panic("pageframes");

	if (k_map(capx, access, frame,
	      0, 512/64, _PAGE_GROWS_UP_FROM_ZERO) )
	then
	{	setframe( frame , dev , bn , capx ,
			access+shortFRAME,
			512/64, 1);
		return( (frame - SUPERD) << 13 );
	}
	else	
	{	u.u_error = EMAP;
		captab[capx].c_refcount--;
		return( NULL );
	}
}

/* name:
	brelse

function:
	to relinquish a block of disk from some superD frame

algorithm:
	free the page frame it is in;
	decrement the reference count on the capability

parameters:
	an address of a page in my virtual space

returns:
	a setting of u.u_error

globals:
	captab		to mark the page capability inactive
	frametab	to find the segment and free the page


calls:

called by:
	everyone

history:
	Designed and coded by Mark Kampe
*/
brelse( addr )
 int addr;
{	register int capx;
	register int frame;

	frame = ((addr >> 13) & 7) + SUPERD;
	capx = _segcap[ frame ];
	if (capx < 0)
	then	return(u.u_error = EMAP);

	frametab[ frame ].f_refcount--;
	captab[ capx ].c_refcount--;
	return;
}

/* name:
	bclear

function:
	to zero a short segment of disk 

algorithm:
	bfetch in the segment;
	clear it;
	brelse the segment;

parameters:
	device id
	physical block number

returns:
	a setting of u.u_error

globals:

calls:
	bfetch
	brelse

called by:
	alloc
	exec

history:
	Designed and coded by Mark Kampe, 11/76
*/

bclear( dev , bn )
 int dev;
 int bn;
{	register int *p;
	register int *p1;

	p = bfetch( dev , bn , _READ_WRITE_ACCESS );
	if (u.u_error)
	then	return;

	p1 = p + 256;
	while( p < p1 )
		*p++ = 0;

	p = p1 - 256;
	brelse( p );
	return;
}
/* name:
	segfetch

function:
	to bring a large segment into core

algorithm:
	obtain a capability for the segment
	map it in

parameters:
	id0	high order segment name
	id1	low order segment name (if both zero,that means allocate one)
	access	desired access
	frax	frame number it should be mapped into (if -1, any SUPERD)
	length	length in bytes segment is desired to have
	dir	direction of growth

returns:
	framex	index of frame it has been mapped into and
	a setting of u.u_error.  if nonzero, returned framex is zero

globals:
	segcap		to check on reasonableness of a specified frame

calls:
	getcap
	k_map
	getpageframe
	setframe

called by:
	exec
	grow
	uinit

history:
	Designed and coded by Mark Kampe, 12/76
*/

segfetch( id0, id1, access, framex, length, dir )
 int id0, id1, access, framex, length, dir;
{	register int capx;
	register int frame;
	register int modes;

	/* make sure the specified frame (if any is reasonable) */
	if (framex == -1) then
	{	frame = getpageframe( SUPERD );
		if (frame < 0) 
		then	panic("pageframes");
	}
	else
	{	frame = framex;
		if (frametab[ frame ].f_refcount != 0)
		then
		{	u.u_error = EBUSY;
			return( NULL );
		}
	}

	/* try to get a capability for the desired segment */
	if ((id0|id1) == 0)
	then	capx = segalloc();	/* if he wants a new segment */
	else	capx = getcap( id0, id1, _LARGE_SEGMENT, access );
	if (capx < 0) then
	{	u.u_error = ENOCAP;
		return( NULL );
	}

	id0 = captab[ capx ].c_id[0];
	id1 = captab[ capx ].c_id[1];

	/* try to map it in, as he wanted */
	length =+ 63;
	length =/ 64;
	if (k_map(capx, access, frame, 0, length, dir)) then
	{	modes = access;
		if (dir == _PAGE_GROWS_DOWN_TO_ZERO)
		then	modes =| downFRAME;
		setframe( frame, id0, id1, capx, modes, length, 1 );
		return( frame );
	}
	else
	{	u.u_error = EMAP;
		captab[capx].c_refcount--;
		return(NULL);
	}
}
/* name:
	segalloc

function:
	to obtain a new scratch segment

algorithm:
	If I have an unused capability for a large segment around
		clear that segment
		return the capability index
	else
		Allocate a new one from the file process
		return its capability index

parameters:
	none

returns:
	capability index or -1

globals:
	captab

calls:
	getcap

called by:
	segfetch

history:
	Designed and coded by Mark Kampe
*/
segalloc()
{	register int capx;
	register struct captab *cp;

	for(capx = 0; capx < _NUM_CLIST_ENTRIES; capx++)
	{	cp = &captab[ capx ];
		if ((cp->c_type == _LARGE_SEGMENT) &&
		    (cp->c_refcount == 0) &&
		    (cp->c_access == _READ_WRITE_ACCESS) ) then
		{	cp->c_refcount++;
			goto gotsegcap;
		}
	}

	/* we have to allocate a new one */
	capx = getcap( 0, 0, _LARGE_SEGMENT, _READ_WRITE_ACCESS );
	if (capx < 0)
		then	return( capx );
	cp = &captab[ capx ];
gotsegcap:
	segclear( cp->c_id[0], cp->c_id[1] );
	return( capx );
}

/* name:
	segrelse

function:
	to remove a large segment from some page frame

algorithm:
	find the capability associated with that segment
	decrement the reference count on the capability.

parameters:
	frame number

returns:
	a setting of u.u_error

globals:
	captab
	frametab
	_segcap

calls:

called by:
	everyone

history:
	Designed and coded by Mark Kampe
*/
segrelse( framex )
 int framex;
{	register int frame;
	register int capx;
	register int count;

	frame = framex;
	capx = _segcap[ framex ];

	if (capx < 0) then
		return( u.u_error = EMAP );

	captab[ capx ].c_refcount--;
	frametab[ framex ].f_refcount--;
}


/* name:
	segclear

function:
	to clear a long segment

algorithm:
	segfetch the segment in with read/write access
	clear it
	release it

parameters:
	id0	high order segment id
	id1	lo order segment id

returns:
	a setting of u.u_error

globals:

calls:
	segfetch
	brelse

called by:
	uinit
	exec

history:
	Designed and coded by Mark Kampe, 12/76
*/
segclear( id0, id1 )
 int id0, id1;
{	register int fx;
	register int *fp;
	register int *endp;

	fx = segfetch( id0, id1, _READ_WRITE_ACCESS, -1, 8192,
			_PAGE_GROWS_UP_FROM_ZERO );
	if (u.u_error)
		then	return(NULL);

	fp = (fx - SUPERD) << 13;
	endp = fp + 4096;

	while( fp < endp )
		*fp++ = 0;

	segrelse( fx );
	return;
}

/*
	getpageframe

function:
	to allocate a frame of virtual space

algorithm:
	Starting in the specified block of segments
		search for an empty one.
	if it is found, return its index.
	if not, return -1;

parameters:
	initial index (USERI, USERD, SUPERI or SUPERD)

returns:
	-1	unable to allocate frame
	number	index of a free frame

globals:
	frametab

calls:

called by:
	bfetch

history:
	Designed and coded by Mark Kampe, 11/76
*/
getpageframe( starting )
 int starting;
{	register int f;
	register int limit;

	f = starting;
	if ((f&7) || (f>24))
	then	return( -1);

	limit = f + 8;
	while( f < limit )
		if (frametab[ f ].f_refcount == 0)
		then	return(f);
		else	f++;

	return( -1 );
}

/* name:
	seginit

function:
	to initialize the page frame table

algorithm:
	( k_init_cap has already initialized segcap )
	for each frame of virtual space
		if k_initcap says there is a segment there,
			init the entry to describe the segment.
			else say the frame is free.
	say that the frames containing the comm and arg blocks
			are not free.

parameters:

returns:

globals:
	frametab
	_segcap

calls:
	setframe

called by:
	main

history:
	Designed and coded by Mark Kampe, 11/76
*/

seginit()
{	register int framex;
	register int capx;

	u.u_sep = _u_sep;

	/* now fill in all the frame table entries */
	for( framex = 0; framex < 32; framex++)
	{	capx = _segcap[ framex ];
		if (capx < 0) then
		{	setframe( framex, 0, 0, -1, 0, 0, 0);
			continue;
		}

		setframe( framex, -1, -1, capx,
			  _READ_ACCESS, 8192/64, 1 );
/* FIX seginit to put the segment IDs of initial segments into frametab */
		if (framex&DATASEGMENT)
		then	frametab[framex].f_modes =| _WRITE_ACCESS;
	}

	/* kluge up the entries for the comm segment and arg segment */
	setframe( SUPERI+7, -1, -1, -2, 0, 0, 1 );
	setframe( SUPERD+6, -1, -1, -2, 0, 0, 1 );
	setframe( SUPERD+7, -1, -1, -2, 0, 0, 1 );
}
/* name:
	setframe

function:
	to fill in an entry in the page frame table

algorithm:
	put the specified values into the specified frame table entry

parameters:
	framex	index into the frame table
	id0	major designation of segment
	id1	minor designation of segment
	capx	associated capability index
	modes	associated modes (and accesses)
	length	length to set it to
	count	reference count for it

returns:

globals:
	frametab

calls:

called by:
	bfetch
	brelse
	seginit

history:
	Designed and coded by Mark Kampe (mostly to reduce ammt of inline code)
*/

setframe( framex, id0, id1, capx, modes, length, count )
{	register struct frametab *fp;

	fp = &frametab[framex];
	fp->f_id[0] = id0;
	fp->f_id[1] = id1;
	fp->f_modes = modes;
	fp->f_length = length;
	fp->f_refcount = count;
	_segcap[framex] = capx;
}
