/*
 * p/n:
 * Program name:      Kernel Dump
 * Author:            Ken Bowler
 * Backup:
 *
 * Description:
 *    This is the main part of the routine to write a complete
 *    core image to the file "dump".
 */
#include "../h/ioconf.h"
#include "../h/param.h"
#include "../h/reg.h"
#include "../h/dir.h"
#include "../h/inode.h"
#include "../h/io.h"
#include "../h/ino.h"
#include "../h/user.h"
#include "../h/dasd.h"

/*
 * Define ccw op codes
 */

#define SEEK     0x07
#define SETSECT  0x23
#define SRCHIDEQ 0x31
#define TIC      0x08
#define RDDATA   0x06
#define WRTDATA  0x05

#define TRKSKP   1      /* number of tracks not used at start of disk */
#define BLKSIZ   4096
#define NBUFF    2      /* number of buffers  - must be at least 2 */

#define LOCK     0      /* buffer locked */

#define LOCZERO  (*(int *) 0)   /* this is very kludgey. */
				/* if a dump is successfully taken, */
				/* this location is set to 0x000D, */
				/* otherwise it is zero */
/*
 * memdump - dumps storage to the file 'dump' in the event of an au crash.
 *           It tries to disturb things as little as possible.
 *           Returns 0 if things worked ok, 1 otherwise.
 */

/*
 * Structure of psa copy
 */
struct  dmppsa {
        char   psa[216];    /* place to copy first part of psa */
        cpu_t  cput;        /* place to store cpu timer */
        cpu_t  clkcmp;      /* place to store clock comparator */
        char   mck[24];     /* copy of machine check stuff */
        long   psw;         /* psw at interrupt */
        tod_t  clock;       /* it should go somewhere */
        char   fixdl[44];   /* rest of fixed logout area */
        long   fpregs[4];   /* place to store floating point registers */
        int    regs[16];    /* place to store registers */
        int    ctlregs[16]; /* place to store control registers */
        char   rest[3584];  /* the rest of it */
};

char    dmpbuffs[BLKSIZ*NBUFF];      /* buffer for disk i/o */


struct  bfinfo {               /* buffer descruptor vector */
        int    bflag;           /* availability bflag */
        int    curblock;        /* current contents block number */
        char   *buffer;         /* address of the buffer */
} buffinfo[NBUFF];


/*
 * An entry in the dasd driver provides us with the following
 */

struct  dmpdisk {            /* dump disk descriptor         */
        int dd_address;      /* disk address                 */
        int dd_blktrk;       /* number blocks/track          */
        int dd_trkcyl;       /* number of tracks/cylinder    */
        int dd_const;        /* stuff needed to calculate    */
        int dd_factor;       /* sector number for set sector */
        int dd_divsor;       /* ccws                         */
} dmpdisk;

struct  dmpcprog {           /* channel program to access disk */
        long cp_seek;
        long cp_setsect;
        long cp_search;
        long cp_tic;
        long cp_rdwrt;

        struct dasdadr {     /* where to put disk address */
               short bin;
               short cyl;
               short track;
               char  record;
               char  sector;
               } dmpdadr;
} dmpcprog;


struct inode  dmpinode;      /* place to build usable inode */
char   dmpfile[DIRSIZ] = "dump";


/*
 * Declare local routines that return somthing.
 */
int    memdump(),writeblk(),dmpwrite(),cvtdblk();
char   *readblk(),*dmpread();

/*
 * Declare external variables.
 */
extern int    memlim;
extern dev_t  dumpdev;         /* the disk where the dump file resides */
struct dasd   dasds[];

/*
 * Declare external routines that return someting.
 */
cpu_t  a_stpt();
tod_t  a_stck();
cpu_t  _a_stckc();
int    fnddump(),dmpio(),dmpioini(),dmpgdesc();

/*
 * Finally here it is.
 */
int    memdump()
{

        char   *address;        /* address currently being dumped */
        struct bfinfo *bi;      /* pointer to buffer info block */
        char   *bp;             /* buffer pointer */
        ccw_t  *ccwp;           /* a pointer to a ccw */
        int    dirsize,srchsize;/* size of directory for dumpfile search */
        struct dinode *di;      /* a pointer to an disk type inode */
        int    dskblk,offset;   /* block # and offset of dumpfile inode */
        tod_t  dumptime;        /* time of dump */
        int    i;               /* a handy counter */
        int    inumber;         /* inode # of dump file */
        int    mpages,fpages;   /* used to give messages on dump size */
        int    pages,savepages; /* used to keep track of pages dumped/to dump */
        struct dmppsa *psa;     /* pointer to psa copy */
	static int    beenhere = 0;    /* check for recursive dumps */

	if (beenhere) {
		printf("dump recursion\n");
		return;
	} else
		beenhere = 1;
	LOCZERO = 0;
	printf("taking a dump\n");
        /*
         * First initialize buffer info blocks
         */
        address = dmpbuffs;     /* point at start of buffer pool */
        for (i = 0; i < NBUFF; i++){
                bi = &buffinfo[i];
                bi->bflag = 1;                 /* mark available */
                bi->curblock = 0;              /* contents undefined */
                bi->buffer = address;          /* set address */
                address += BLKSIZ;             /* increment address */
        }
        bi = &buffinfo[0];              /* use first buffer for psa */
        bi->bflag = LOCK;
        psa = (struct dmppsa *)bi->buffer;
        /*
         * Copy the psa so we dump an pristine copy.
         */
        bcopy((caddr_t) 0, (caddr_t) psa, BLKSIZ);
        /*
         * Now make like a store status was done, so volitile things are saved.
         */
        psa->cput = a_stpt();          /* store cpu timer */
        psa->clkcmp = _a_stckc();       /* store the clock comparator */
        psa->psw = *(long *) &u.u_ar0[RPS];
        /*
         * no area defined for clock; we put it just after psw
         */
        psa->clock = a_stck();         /* store current clock */
        dumptime = psa->clock;         /* save dump time */

        bcopy((caddr_t) &u.u_ar0[FR0], (caddr_t) psa->fpregs, 32);
        bcopy((caddr_t) u.u_ar0, (caddr_t) psa->regs,64);
        _a_stctl(psa->ctlregs);       /* store control registers */
        /*
         * Find dasd descriptor for dump device.
         */
        if (dmpgdesc(&dmpdisk, dumpdev) != 0) {
                printf("Dumper: unable to get dump disk descriptor.\n");
                return;
        }
        /*
         * Now find dump directory inode. We read from the disk, in case its
         * corruption was what caused our demise.
         */
        printf("dump dev %x\n", dmpdisk.dd_address);
        if (dmpioini(dmpdisk.dd_address) != 0) {        /* initialize for i/o */
                printf("Dumper: unable to do I/O on dump disk.\n");
                return;
        }

        ccwp = (ccw_t *) &dmpcprog;   /* initialize the channel program */
        ccwp->cc_dblw  = 0;
        ccwp->cc_cmd   = SEEK;
        ccwp->cc_addr  = (int)&dmpcprog.dmpdadr.bin;
        ccwp->cc_cc    = 1;
        ccwp->cc_count = 6;
        ccwp++;
        ccwp->cc_dblw  = 0;
        ccwp->cc_cmd   = SETSECT;
        ccwp->cc_addr  = (int)&dmpcprog.dmpdadr.sector;
        ccwp->cc_cc    = 1;
        ccwp->cc_count = 1;
        ccwp++;
        ccwp->cc_dblw  = 0;
        ccwp->cc_cmd   = SRCHIDEQ;
        ccwp->cc_addr  = (int)&dmpcprog.dmpdadr.cyl;
        ccwp->cc_cc    = 1;
        ccwp->cc_count = 5;
        ccwp++;
        ccwp->cc_dblw  = 0;
        ccwp->cc_cmd   = TIC;
        ccwp->cc_addr  = (int)&dmpcprog.cp_search;
        ccwp++;
        ccwp->cc_dblw  = 0;
        ccwp->cc_cmd   = RDDATA;
        ccwp->cc_addr  = 0;
        ccwp->cc_cc    = 0;        /* this is the end */
        ccwp->cc_count = BLKSIZ;

        if ((bp = dmpread(2)) == NULL) {   /* read first inode block */
                printf("Dumper: unable read base inode block.\n");
                return;
        }
        /*
         * Get the inode for dump directory, inode # 2.
         */
	di = (struct dinode *) bp;
        di++;                       /* point at dinode #2 */
        expinode(di);               /* convert inode to usable form */
        inumber = 0;
        dirsize = dmpinode.i_size;
        for (i = 0; i < 10; i++){
                if ((dirsize < 0) || ((bp = readblk(i)) == NULL)) {
                        printf("Dumper: dump file '%s' not found.\n", dmpfile);
                        return;
                }
                if (dirsize < BLKSIZ)
                        srchsize = dirsize;
                else
                        srchsize = BLKSIZ;
                inumber = fnddump(dmpfile, (struct direct *) bp, srchsize);
                if (inumber != 0)
			break;
                dirsize = dirsize - BLKSIZ;
        }

        /*
         * The dump file exists, read in its inode.
         */
        inumber -= 1;                           /* decrement indumber */
        dskblk = (inumber / INOPB) + 2;         /* calulate inode block */
        offset = (inumber % INOPB);
        if ((bp = dmpread(dskblk)) == NULL){    /* read the inode block */
                printf("Dumper: unable to read inode block for dump file '%s'.\n", dmpfile);
                return;
        }
	di = (struct dinode *) bp;
        di += offset;           /* set pointer to the disk type inode */
        if (di->di_size > memlim)
                pages = (memlim + 1) / 4096;
        else
                pages =  di->di_size / 4096;
        savepages = pages;    /* save the number of pages to dump */
        if (pages < 1) {
                printf("Dumper: dump file '%s' is too small to dump anything.\n", dmpfile);
                return;
        }

        /*
         * Now build a usable inode form the one read from disk.
         */
        expinode(di);

        if (writeblk(0, (char *) psa) != 0) {
               printf("Dumper: permanent I/O error writing dump file '%s'.\n", dmpfile);
               return;
        }
        bi->bflag = 1;               /* release buffer for other use */
        /*
         * Finally we dump out rest of memory up to "pages".
         */
        address = (char *) 4096;
        for (i=1; i < pages; i++){
                if (writeblk(i, address) != 0) {
                       printf("Dumper: permanent I/O error writing dump file '%s'.\n", dmpfile);
                       return;
                }
                address += 4096; /* increment dump address */
        }
        mpages = (memlim + 1) / 4096;     /* number of pages in system */
        fpages = dmpinode.i_size / 4096;    /* number of pages in file */

        if (savepages < mpages)
                printf("Dumper: dump file '%s' is too small; %d pages dumped of %d.\n",
                        dmpfile, savepages, mpages);
        else if (savepages < fpages){
                        pages = fpages - savepages;
                        printf("Dumper: dump file '%s' is %d pages longer than %d page dump.\n",
                                dmpfile, pages, savepages);
                }
                else   printf("Dumper: %d pages dumped on dump file '%s'.\n", savepages, dmpfile);
        if ((bp = dmpread(dskblk)) == NULL) {    /* get inode block again */
                printf("Dumper: unable to update dump file '%s' modification time.\n",
                       dmpfile);
                return;
        }
        di = (struct dinode *) bp;
        di += offset;                     /* point at dump file inode */
        di->di_mtime = ctime(dumptime);   /* update modification time */
        if (dmpwrite(dskblk,bp) != 0)
                printf("Dumper: unable to update dump file '%s' modification time.\n",
                       dmpfile);
	printf("dump complete\n");
	LOCZERO = 0x000d;
        return;
}


/*
 * expinode - convert inode from disk format to a more usable format.
 */
expinode(di)
struct dinode *di;
{
        char   *p1,*p2;
        int    i;

        dmpinode.i_mode = di->di_mode;
        dmpinode.i_nlink = di->di_nlink;
        dmpinode.i_uid = di->di_uid;
        dmpinode.i_gid = di->di_gid;
        dmpinode.i_size = di->di_size;
        p1 = (char *)dmpinode.i_un.i_addr;
        p2 = (char *)di->di_addr;
        for(i=0; i<NADDR; i++) {
                *p1++ = 0;
                *p1++ = *p2++;
                *p1++ = *p2++;
                *p1++ = *p2++;
        }
}


/*
 * readblk - routine to read a block for a file given relative block
 *           number. Inode for file in dmpinode.
 *         - Returns buffer address if successful, NULL otherwise.
 */
char    *readblk(relblk)
int     relblk;       /* the relative block number to read */
{

        int dskblk;   /* disk block number */

        /*
         * Convert relative block number to a disk address.
         */
        dskblk = cvtdblk(relblk);
        if (dskblk == 0)
		return(NULL);
        return(dmpread(dskblk));
}


/*
 * writeblk - routine to write a block for a file given relative block
 *            number. Inode for file in dmpinode.
 *          - Returns 0 if write succesful, 1 otherwise.
 */
int     writeblk(relblk,buffer)
int     relblk;       /* the relative block number to read */
char    *buffer;      /* block to write */
{

        int dskblk;  /* disk block number */

        /*
         * Convert relative block number to a disk address.
         */
        dskblk = cvtdblk(relblk);
        if (dskblk == 0) return(1);
        return(dmpwrite(dskblk,buffer));
}


/*
 * dmpread - routine to read a given disk block into a buffer.
 *         - Returns buffer address if successful, NULL otherwise.
 */
char     *dmpread(blkno)
int      blkno;       /* the disk block number to read */
{
        int    i,max,point;
        struct bfinfo *bip; /* buffer info block pointer */
	ccw_t   *ccwp;

        max = 0;
        point = 0;
        for (i = 0; i < NBUFF; i++){
                bip = &buffinfo[i];  /* point at buffer info block */
                if (bip->bflag != LOCK){
                        if(max < bip->bflag){
                                max = bip->bflag;    /* find oldest buffer */
                                point = i;     /* save pointer to oldest */
                        }
                        bip->bflag += 1;        /* age the buffer */
                }
                if (bip->curblock == blkno) break;
        }
        bip = &buffinfo[point];     /* point at the one we found */
        bip->bflag = 1;                /* make this lru */

        if (blkno == bip->curblock) return(bip->buffer);
        blk_adr(blkno);           /* set up disk address */
	ccwp = (ccw_t *) &dmpcprog.cp_rdwrt;
        ccwp->cc_cmd = RDDATA;
        ccwp->cc_addr = (int) bip->buffer;
        bip->curblock = blkno;         /* save this block number */
        if (dmpio(dmpdisk.dd_address,&dmpcprog) == 0)
                return(bip->buffer);
        else
                return(NULL);
}


/*
 * dmpwrite - routine to write a given disk block from buffer provied.
 *          - Returns 0 if write succesful, 1 otherwise.
 */
int      dmpwrite(blkno,buffer)
int      blkno;       /* the block number to read */
char     *buffer;     /* address of buffer */
{
	ccw_t   *ccwp;

        blk_adr(blkno);           /* set up disk address */
	ccwp = (ccw_t *) &dmpcprog.cp_rdwrt;
        ccwp->cc_cmd  = WRTDATA;
        ccwp->cc_addr = (int)buffer;
        return(dmpio(dmpdisk.dd_address,&dmpcprog));
}


/*
 * blk_adr - routine to convert a disk block number into dasd address
 *           and sector number.
 */

blk_adr(blkno)
int blkno;
{
        struct  dasdadr *addr;
        int     rec,tmp;

        addr = &dmpcprog.dmpdadr;

        rec = (blkno % dmpdisk.dd_blktrk) + 1;          /* record id */
        tmp = (blkno / dmpdisk.dd_blktrk) + TRKSKP;     /* track number on disk */
        addr->cyl = tmp / dmpdisk.dd_trkcyl;            /* cylinde number */
        addr->track = tmp % dmpdisk.dd_trkcyl;          /* track number in cylinder */
        addr->record = rec;
        addr->sector = (dmpdisk.dd_const + (rec-1)*dmpdisk.dd_factor) / dmpdisk.dd_divsor; /* RPS */
}
 

/*
 * cvtdblk - Routine to convert a relative block number into a disk
 *           block number.
 *         - Returns disk block number, or 0 if invalid.

 */

int cvtdblk(relblk)
int     relblk;
{
        struct inode *ip;
        int i,j,sh;
        int nb, *bap;

        ip = &dmpinode;
        if(relblk < 0) return(0);
	/*
	 * blocks 0..NADDR-4 are direct blocks
	 */
	if(relblk < NADDR-3) {
                nb = ip->i_un.i_addr[relblk];
                return(nb);
        }

	/*
	 * addresses NADDR-3, NADDR-2, and NADDR-1
	 * have single, double, triple indirect blocks.
	 * the first step is to determine
	 * how many levels of indirection.
	 */
	sh = 0;
	nb = 1;
        relblk -= NADDR-3;
	for(j=3; j>0; j--) {
                sh +=  NSHIFT;
                nb <<= NSHIFT;
                if(relblk < nb) break;
		relblk -= nb;
	}
        if(j == 0) return(0);         /* too bad, not valid */

	/*
	 * fetch the address from the inode
	 */
	nb = ip->i_un.i_addr[NADDR-j];
        if(nb == 0) return(0);        /* indirect block is absent */

	/*
	 * fetch through the indirect blocks
	 */
        for(; j <= 3 ; j++) {
                if ((bap = (int *) dmpread(nb)) == NULL)
			return(0);
                sh -= NSHIFT;
		i = (relblk>>sh) & NMASK;
                nb = bap[i];
                if(nb == 0) return(0);
        }
        return(nb);
}
/*
 * Routine to return information about a disk for the panic dumper
 *   This routine should really exist elswhere, but the dasd structure
 *   and the dskdesc structure are not header files as they should be,
 *   and i don't want to straighten this all up now. KHB.
 */

dmpgdesc(dmp, dev)
register struct dmpdisk *dmp;
dev_t   dev;
{
        register struct dasd *d;
        register struct dskdesc *dd;

        d = &dasds[major(dev)];
        if(d->d_desc == NULL) return(1);  /* I reaaly don't wan't to find out */
        dmp->dd_address = d->d_devaddr;
        dd = d->d_desc;
        dmp->dd_blktrk = dd->d_blktrk;
        dmp->dd_trkcyl = dd->d_trkcyl;
        dmp->dd_const  = dd->d_const;
        dmp->dd_factor = dd->d_factor;
        dmp->dd_divsor = dd->d_divsor;
        return(0);
}
/*
 *  fnddump - find the dump file in a directory
 */
fnddump(file, addr, len)
char    *file;
struct  direct  *addr;
int     len;
{
	int     i;

        while (len > 0) {

		for (i = 0; i < DIRSIZ; i++) {
			if (file[i] != addr->d_name[i])
				break;
			if (file[i] == '\0')
				return(addr->d_ino);
		}
                if (i == DIRSIZ)
                        return(addr->d_ino);
		addr++;
		len -= sizeof(struct direct);
	}
	return(0);
}
