#include "../h/param.h"
#include "../h/proc.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/systm.h"
#include "../h/370.h"
#include "../h/text.h"

/*
 * This file contains all the "low-level" memory management routines.
 */

#define pntopa(x) ((caddr_t) ((x) << 12))     /* page number to page addr */
#define patopn(x) ((((int) x) >> 12) & 0xfff) /* page addr to page number */

#define PPAGE (PZERO-1)
 
#define SEGLEN  256                     /* length of segment tables */
#define PTLEN   16                      /* length of page tables */
#define CR1BITS ((SEGLEN/16 - 1) << 24) /* segment table length bits for cr1 */

union segtab {                          /* describes segment translation table */
	union sg_ent {
                struct {
                        unsigned  sg_length  :  4;  /* length of segment table */
                        unsigned             :  4;
                        unsigned  sg_pagetab : 24;  /* addr of page table */
                };
                struct {
                        unsigned             : 31;
                        unsigned  sg_invalid :  1;  /* segment invalid bit */
                };
	} sg[SEGLEN];
	union segtab *sg_next;
};

struct pagetab {            /* describes page translation table */
	struct  pagetab *pt_next;
	int     pt_junk;
        struct pt_ent {
                unsigned short  pt_number : 12;  /* high order bits of page addr */
                unsigned short  pt_invalid:  1;  /* page invalid bit */
        } pt_entry[PTLEN];
};

struct coretab {
	char c_storkey;   /* storage key */
	short c_nxtfpag;  /* next free page number */
};

struct coretab *coretab;   /* initalized in zero.s */

int ffpage;                        /* coretab index of first free page */
struct pagetab *ffptab = NULL;     /* first free page table */
union segtab *safesegptr = NULL;   /* points to list of available segment tables */
union segtab *unsafesegptr = NULL; /* points to list of used segment tables */


/*
 * Paginit is called during system initialization;
 * it sets up a table of available main memory pages.
 * The pointer coretab has previously been set by zero.s to point to the
 * first free address; the table coretab will start at this address.
 */
paginit()
{
	register int i, good, bad, addr;
	extern char etext;
	caddr_t p;

        /*
	 * binary search for highest addressable page
         */
	good = 0;
	bad = 4096;              /* first page past 16 meg */
	while (good + 1 < bad) {
		addr = (good + bad) / 2;
		if (tryaddr(pntopa(addr)))
			good = addr;
		else
			bad = addr;
	}
        /*
         * Pages in low memory which are not free include those with
         * addresses lower than the pointer coretab and the page(s)
         * occupied by the coretab table.
         */
	ffpage = btoc(((int) coretab) + (sizeof (struct coretab) * good));
	for (i = 0; i < ffpage; i++)
		coretab[i].c_storkey = INITKEY;
	for (; i < good; i++) {
		coretab[i].c_storkey = INITKEY;
		coretab[i].c_nxtfpag = i + 1;
		if (vm)
			d_relpage(pntopa(i));
	}
	coretab[good].c_storkey = INITKEY;
	coretab[good].c_nxtfpag = 0;
	if (vm)
		d_relpage(pntopa(good));
	memlim = (int) pntopa(good + 1) - 1;
	pagecnt = maxmem = good - ffpage + 1;

	a_lctl(0, 0, CR0);
        a_ptlb();
	/*
	 * Protect the kernel text and data pages.
	 */
	p = (caddr_t) 0;
	a_ssk2k(p, USERKEY);   /* Leave writable area for zero.s */
	p += PSIZE/2;
	a_ssk2k(p, TEXTKEY);   /* Protect zero.s code */
	p += PSIZE/2;
	for (; p < &etext; p += PSIZE)
		ssk(p, TEXTKEY);
	for (; p < pntopa(ffpage); p += PSIZE)
		ssk(p, USERKEY);
}

/*
 * The next few routines all deal with segment tables.
 */

/*
 * Allocate a segment table.
 * If there are no more safe tables, then
 * if there are tables to recycle, do a ptlb
 * and recycle them,
 * if not, get a new page and break it up.
 */
union segtab *
getstab()
{
	register union segtab *p, *q;

	if (safesegptr == NULL)
	        if (unsafesegptr != NULL) {       /* recover bad list */
		        safesegptr = unsafesegptr;
		        unsafesegptr = NULL;
		        a_ptlb();
	        } else {                  /* get new ones for good list */
		        p = (union segtab *) getpage();
			q = p + PSIZE / sizeof (*p);
		        for (; p < q; p++) {
			        p->sg_next = safesegptr;
			        safesegptr = p;
		        }
	        }
	p = safesegptr;
	safesegptr = p->sg_next;
	p->sg_next = (union segtab *) 0;  /* hardware wants bits 4-7 cleared */
	return(p);
}

/*
 * Put a segment table back on the free list.
 */
frestab(s)
register union segtab *s;
{
	s->sg_next = unsafesegptr;
	unsafesegptr = s;
}

/*
 * Place user's page information in segment table.
 */
mkstab(up, flag)
struct user *up;
int flag;           /* 1 = new segtab needed */
{
	register union segtab *s;
	register struct pagetab *pt;
	register int i, j;
	union segtab *old_segtab;

	if((s = up->u_segtab) == NULL)
		s = up->u_segtab = getstab();
	else if (flag) {
		frestab(old_segtab = up->u_segtab);
		if ((s = up->u_segtab = getstab()) != old_segtab)
			up->u_chreg = 1;
	}
	/*
	 * Text and data segments.
	 */
	i = 0;
	for (pt = up->u_tseg.u_pages; pt != NULL; pt = pt->pt_next) {
		s->sg[i].sg_length = PTLEN - 1;
		s->sg[i].sg_pagetab = (unsigned) pt->pt_entry;
		i++;
	}
	for (pt = up->u_dseg.u_pages; pt != NULL; pt = pt->pt_next) {
		s->sg[i].sg_length = PTLEN - 1;
		s->sg[i].sg_pagetab = (unsigned) pt->pt_entry;
		i++;
	}
	/*
	 * Stack segment.
	 */
	j = SEGLEN - 1;
	for (pt = up->u_sseg.u_pages; pt != NULL; pt = pt->pt_next) {
		s->sg[j].sg_length = PTLEN - 1;
		s->sg[j].sg_pagetab = (unsigned) pt->pt_entry;
		j--;
	}
	/*
	 * Void area.
	 */
	for(; i <= j; i++)
		s->sg[i].sg_invalid = 1;
}

/*
 * The next few routines deal with page tables.
 */

/*
 * Allocate and initialize a page table.
 * If none are immediately available, grab a page and use it.
 */
struct pagetab *
getptab()
{
        register struct pagetab *p, *q;
        register int i;

        if (ffptab == NULL) {
                p = (struct pagetab *) getpage();
                q = p + PSIZE / sizeof (*p);
                for (; p < q; p++) {
                        p->pt_next = ffptab;
		        ffptab = p;
		}
        }
        p = ffptab;
        ffptab = ffptab->pt_next;
        p->pt_next = NULL;
        for(i = 0; i < PTLEN; i++)
		p->pt_entry[i].pt_invalid = 1;
        return(p);
}
 
/*
 * Put a page table on the free list
 */
freptab(p)
struct pagetab *p;
{
        p->pt_next = ffptab;
        ffptab = p;
}

/*
 * The next few routines deal with strings of pages.
 */

/*
 * Add diff pages to the string of data pages.
 */
sbreak1(diff, pgtb)
int     diff;
struct  u_pgtb *pgtb;
{
	register struct  pagetab *pt;
	register i, npages;
	caddr_t p;

	if (diff > 0) {                 /* add pages */
		npages = pgtb->u_size;
		pgtb->u_size += diff;
		if (pgtb->u_pages == NULL) {
			pgtb->u_pages = pt = getptab();
			i = 0;
		} else {
			pt = pgtb->u_pages;
			while (npages > PTLEN) {
				npages -= PTLEN;
				pt = pt->pt_next;
			}
			if (npages == PTLEN) {
				pt->pt_next = getptab();
				pt = pt->pt_next;
				i = 0;
			} else
				i = npages;
	        }
		/*
		 * At this point, pt->pt_entry[i] is available for inserting a new page.
		 */
		do {
			for (; i < PTLEN && diff--; i++) {
				p = getpage();
				pt->pt_entry[i].pt_number = patopn(p);
				pt->pt_entry[i].pt_invalid = 0;
			}
			if (diff > 0) {
				pt->pt_next = getptab();
				pt = pt->pt_next;
				i = 0;
			}
		} while (diff > 0);
		pt->pt_next = NULL;
	} else if (diff < 0) {          /* subtract pages */
		npages = pgtb->u_size += diff;  /* npages = number of pages to keep */
		if (pgtb->u_pages == NULL)
			panic("sbreak1 - no pages\n");
		pt = pgtb->u_pages;
		while (npages > PTLEN) {
			npages -= PTLEN;
			if ((pt = pt->pt_next) == NULL)
				panic("sbreak1 - too few pages\n");
		}
		for (i = npages; i < PTLEN; i++)
			if (!pt->pt_entry[i].pt_invalid) {
				freepag(pntopa(pt->pt_entry[i].pt_number));
				pt->pt_entry[i].pt_invalid = 1;
			}
		freepgs(pt->pt_next);
		pt->pt_next = NULL;
	}
}

/*
 * Add diff pages to the string of stack pages.
 */
grow1(diff, pgtb)
int     diff;
struct  u_pgtb *pgtb;
{
	register struct pagetab *pt;
	register i, npages;
	caddr_t p;

	if (diff <= 0)
		return;
	if (pgtb->u_pages == NULL) {
		pt = pgtb->u_pages = getptab();
		i = PTLEN - 1;
	} else {
		pt = pgtb->u_pages;
		npages = pgtb->u_size;
		while (npages > PTLEN) {
			npages -= PTLEN;
			pt = pt->pt_next;
		}
		if (npages == PTLEN) {
			pt->pt_next = getptab();
			pt = pt->pt_next;
			i = PTLEN - 1;
		} else
			i = PTLEN - 1 - npages;
	}
	pgtb->u_size += diff;
	/*
	 * At this point, pt->pt_entry[i] is available for inserting a new page.
	 */
	do {
		for (; i >= 0 && diff--; i--) {
			p = getpage();
                        pt->pt_entry[i].pt_number = patopn(p);
                        pt->pt_entry[i].pt_invalid = 0;
		}
		if (diff > 0) {
			pt->pt_next = getptab();
			pt = pt->pt_next;
			i = PTLEN - 1;
		}
	} while (diff > 0);
	pt->pt_next = NULL;
}

/*
 * Copy the data from the page string from_pgtb to the string to_pgtb.
 */
copysgs(from_pgtb, to_pgtb)
struct u_pgtb *from_pgtb, *to_pgtb;
{
	register i;
        register struct pagetab *from_pt, *to_pt;

	from_pt = from_pgtb->u_pages;
	to_pt = to_pgtb->u_pages;
	while (from_pt) {
		if (to_pt == NULL)
			panic("copysgs -- zero to_pt\n");
		for (i = 0; i < PTLEN; i++)
			if (!from_pt->pt_entry[i].pt_invalid)
                                copypag(pntopa(from_pt->pt_entry[i].pt_number),
                                          pntopa(to_pt->pt_entry[i].pt_number));
		from_pt = from_pt->pt_next;
		to_pt = to_pt->pt_next;
	}
}

/*
 * Set the protect key for the string of pages pointed to by pgtb.
 */
protect(pgtb, key)
struct u_pgtb *pgtb;
register key;
{
	register i;
        register struct pagetab *pt;

	pt = pgtb->u_pages;
	while (pt) {
		for (i = 0; i < PTLEN; i++)
			if (!pt->pt_entry[i].pt_invalid)
				ssk(pntopa(pt->pt_entry[i].pt_number), key);
		pt = pt->pt_next;
	}
}

/*
 * Free pages, set count to zero.
 */
freesgs(pgtb)
struct u_pgtb *pgtb;
{
	if (pgtb->u_size)
	        freepgs(pgtb->u_pages);
	pgtb->u_pages = NULL;
	pgtb->u_size = 0;
}

/*
 * Free the pages in the string of segments pointed to by pt.
 */
freepgs(pt)
register struct pagetab *pt;    /* not struct u_pgtb */
{
	register i;
	register struct pagetab *tpt;

	while (pt) {
		for (i = 0; i < PTLEN; i++)
			if (!pt->pt_entry[i].pt_invalid)
				freepag(pntopa(pt->pt_entry[i].pt_number));
		tpt = pt;
		pt = pt->pt_next;
		freptab(tpt);
	}
}

/*
 * The next few routines deal with pages.
 */

/*
 * Allocate a free page
 */
caddr_t
getpage()
{
	register caddr_t p;

	while(pagecnt <= 0)
		sleep((caddr_t)&ffpage, PPAGE);
	p = pntopa(ffpage);
	ffpage = coretab[ffpage].c_nxtfpag;
	pagecnt--;
	ssk(p, USERKEY);
	if (!vm)
		zeropag(p);  /* Vm clears pages we release with diagnose. */
	return(p);
}

/*
 * Put a page back on the free list.
 * Tell VM we freed the page so it can free memory and disk slots.
 */
freepag(p)
register caddr_t p;
{
	if ((int)p % PSIZE)
		panic("freepag - not on page boundary\n");
	if (pagecnt == 0)
		wakeup((caddr_t)&ffpage);
	coretab[patopn(p)].c_nxtfpag = ffpage;
	ffpage = patopn(p);
	if (vm)
	        d_relpage(p);
	pagecnt++;
}

/*
 * Finally, we have miscellaneous memory management routines.
 */

/*
 * If the current key of page p is not key, set it to key.
 */
ssk(p, key)
register caddr_t p;
register key;
{
	if (coretab[patopn(p)].c_storkey != key) {
		a_ssk(p, key);
		coretab[patopn(p)].c_storkey = key;
	}
}

/*
 * Return the storage key associated with a given page
 */
isk(p)
caddr_t p;
{
	return(coretab[patopn(p)].c_storkey);
}

/*
 * Load the control registers for a user process,
 * but only if we really need to
 */
sureg()
{
	static struct user *lastu = NULL;

	if(lastu != &u || u.u_chreg) {
		lastu = &u;
		u.u_chreg = 0;
		a_lctl(1, 1, CR1BITS | (int)u.u_segtab);
		a_lctl(8, 11, u.u_ps.ps_cr8, u.u_ps.ps_cr9, u.u_ps.ps_cr10, u.u_ps.ps_cr11);
	}
}

/*
 * Simulate the 'Load Real Address' instruction
 * This is done so VM doesn't get control each time
 * we need an address translated.
 */

caddr_t
ldreal(vaddr)
caddr_t vaddr;
{
	union sg_ent *sge;
	struct pt_ent *pte;
	union {
                struct {
                        unsigned ad_zero : 8;
                        unsigned ad_sidx : 8;  /* segment table index */
                        unsigned ad_pidx : 4;  /* page table index */
                        unsigned ad_byte : 12; /* byte address within page */
                } ad_virt;
		struct {
			unsigned         : 8;
			unsigned ad_pgno : 12; /* pageno for real addr */
		} ad_real;
		caddr_t ad2;
	} ad;

	ad.ad2 = vaddr;
	sge = &u.u_segtab->sg[ad.ad_sidx];
	if (sge->sg_invalid)
		return((caddr_t) -1);  /* segment translation exception */
        pte = &((struct pt_ent *) sge->sg_pagetab)[ad.ad_pidx];
	if (pte->pt_invalid)
		return((caddr_t) -1);  /* page translation exception */
	ad.ad_pgno = pte->pt_number;
	ad.ad_zero = 0;        /* The poo says lra does this. */
	return(ad.ad2);
}
