/*
 * Driver for DASDs with RPS.
 */
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/370.h"
#include "../h/io.h"
#include "../h/ioconf.h"
#include "../h/dasd.h"

#define DSKPRI (PZERO+1)        /* open priority */
 
struct dasd dasds[NDASD];
/*
 * DASD type table
 * Keep in descending order of # tracks
 */
struct dskdesc dskdesc[NTYPS] = {
        14858/(BSIZE+198),      768,    118,    BSIZE+198,      84,     /*2305*/
        14568/(BSIZE+432),      384,    234,    BSIZE+430,      168,    /*2305*/
        19254/(BSIZE+185),      30,     389,    BSIZE+185,      156,    /*3350*/
        13165/(BSIZE+135),      19,     237,    BSIZE+135,      105,    /*3330*/
        8535/(BSIZE+167),       12,     353,    BSIZE+167,      140,    /*3340*/
};

 
/*
 * Open a DASD device.
 * Determine type, check for writeability, find size, and initialize ccws.
 */
dskopen(dev, flag)
{
        register struct dasd *d;
	struct dskdesc *dd, *dsktype();
        int dsktioi();
	register i;
	ccw_t *ccwp;
 
        d = &dasds[major(dev)];
        while (d->d_init == 1)     /* in case another open or close is in progress */
		sleep((caddr_t)d, DSKPRI);
        if (d->d_init) {
                if (flag && !d->d_rw)
                        u.u_error = EROFS;
                return;
        }
        d->d_init = 1;
        d->d_devaddr = bdevsw[major(dev)].d_baseaddr;
	d->d_devtab = bdevsw[major(dev)].d_tab;
        d->d_done = 0;
        tio(d->d_devaddr, dsktioi, (int)d);
        while (d->d_done == 0)
		sleep((caddr_t)&d->d_done, DSKPRI);
        if (d->d_done != 1) {
                u.u_error = ENXIO;
                d->d_init = 0;
                wakeup((caddr_t)d);
                return;
        }
	d->d_dead = 0;
        if ((dd = dsktype(d)) <= 0) {
                u.u_error = ENXIO;
                d->d_init = 0;
                wakeup((caddr_t)d);
                return;
        }
        d->d_desc = dd;
        if (vm)
		d->d_rw = flag && d_rwdasd(d->d_devaddr);
        else
		d->d_rw = flag;
        if (flag && !d->d_rw) {
                u.u_error = EROFS;
                d->d_init = 0;
                wakeup((caddr_t)d);
                return;
        }
        d->d_maxblk = (dskcyls(d) * dd->d_trkcyl - TRKSKP) * dd->d_blktrk;
	d->d_blkno = 0;
	d->d_dir = FOREWARD;
	/*
	 * Set up skeleton channel programs.
	 * For details, see GA26-1592, p. 70.
	 */
	for (i = 0; i < NCP; i++) {
		ccwp = &d->d_ccws[i][0];
		ccwp->cc_dblw = 0;
	        ccwp->cc_cmd = SEEK;
	        ccwp->cc_addr = (int)&d->d_recloc[i].d_bb;
	        ccwp->cc_cc = 1;
	        ccwp->cc_count = 6;
		ccwp++;
		ccwp->cc_dblw = 0;
	        ccwp->cc_cmd = SETSCTR;
	        ccwp->cc_addr = (int)&d->d_recloc[i].d_sector;
	        ccwp->cc_cc = 1;
	        ccwp->cc_count = 1;
		ccwp++;
		ccwp->cc_dblw = 0;
	        ccwp->cc_cmd = SRCHIDE;
	        ccwp->cc_addr = (int)&d->d_recloc[i].d_cc;
	        ccwp->cc_cc = 1;
	        ccwp->cc_count = 5;
		ccwp++;
		ccwp->cc_dblw = 0;
	        ccwp->cc_cmd = TIC;
	        ccwp->cc_addr = (int)(ccwp-1);
	        ccwp->cc_cc = 1;
	        ccwp++;
	        /* read or write */
		ccwp->cc_dblw = 0;
	        ccwp->cc_count = BSIZE;
	}
        d->d_init = 2;
	iosetup(d->d_devaddr, MINT);
        wakeup((caddr_t)d);
}
 
/*
 * Close DASD.
 * Drain queue and invalidate buffers.
 */
dskclose(dev)
{
	register struct buf *dt;

	dasds[major(dev)].d_init = 1;
	bflush(dev);
	dt = bdevsw[major(dev)].d_tab;
	while (dt->b_actf || dt->b_actl)
		 sleep((caddr_t)dt, PRIBIO);
	binval(dev);
        dasds[major(dev)].d_init = 0;
	wakeup((caddr_t)&dasds[major(dev)]);
}
 
/*
 * DASD strategy routine.
 *
 * The basic strategy is to read/write up to NCP blocks at a time.
 * For efficiency reasons it is desirable to sort the blocks,
 * but they can't just be sorted in order or some might never get
 * done.  So two lists are maintained, one to be used while the
 * disk head is moving outward (head of list is in dt->b_actf,
 * links are in bp->av_forw) and one to be used while the disk
 * head is moving inward (head of list is in dt->b_actl,
 * links are in bp->av_forw).  A third list of blocks which
 * are in process (i.e. an sio has been issued on them) is also
 * maintained (head of list in d->d_iolist, links are in
 * bp->av_forw). These lists are manipulated by insertout,
 * insertin, and getnxtbp.
 */
dskstrat(bp)
struct buf *bp;
{
        register struct dasd *d;
        struct buf *dt;
 
        d = &dasds[major(bp->b_dev)];
        dt = d->d_devtab;
        if (bp->b_blkno < 0 || bp->b_blkno >= d->d_maxblk || d->d_dead) {
                bp->b_flags |= B_ERROR;
                iodone(bp);
                return;
        }
	if (bp->b_blkno > d->d_blkno)
		insertout(bp, dt);
	else
		insertin(bp, dt);
        if (dt->b_active == 0)
                dskstart(d);
}

/*
 * Insert a block in the linked list of records going out.
 */
static
insertout(bp, dt)
register struct  buf *bp;
register struct  buf *dt;
{
	register struct  buf *tp;

	if (dt->b_actf == 0 || bp->b_blkno < dt->b_actf->b_blkno) {
		bp->av_forw = dt->b_actf;
		dt->b_actf = bp;
	} else {
		tp = dt->b_actf;
		while (tp->av_forw && tp->av_forw->b_blkno < bp->b_blkno)
			tp = tp->av_forw;
		bp->av_forw = tp->av_forw;
		tp->av_forw = bp;
	}
}

/*
 * Insert a block in the linked list of records coming in.
 */
static
insertin(bp, dt)
register struct  buf *bp;
register struct  buf *dt;
{
	register struct  buf *tp;

	if (dt->b_actl == 0 || bp->b_blkno > dt->b_actl->b_blkno) {
		bp->av_forw = dt->b_actl;
		dt->b_actl = bp;
	} else {
		tp = dt->b_actl;
		while (tp->av_forw && tp->av_forw->b_blkno > bp->b_blkno)
			tp = tp->av_forw;
		bp->av_forw = tp->av_forw;
		tp->av_forw = bp;
	}
}

/*
 * Set up channel program(s) and start a DASD transfer.
 * Called from both process and interrupt levels.
 */
dskstart(d)
register struct dasd *d;
{
        int dskintr();
	struct buf *dt, *getnxtbp();
	register struct buf *bp;
 
	dt = d->d_devtab;
        dt->b_active = 1;
	d->d_iolist = bp = getnxtbp(d);
	d->d_cpst = d->d_cpend = 0;
	makecprog(bp, d);
	while (++d->d_cpend < NCP && (dt->b_actf || dt->b_actl)) {
		bp->av_forw = getnxtbp(d);
		bp = bp->av_forw;
		makecprog(bp, d);
	}
	bp->av_forw = 0;
	d->d_ccws[d->d_cpend-1][4].cc_cc = 0;
	sio(d->d_devaddr, &d->d_ccws[0][0], dskintr, (int)d);
}

/*
 * Get the next block from the queues set up by dskstrat.
 */
static struct buf *
getnxtbp(d)
register struct  dasd *d;
{
	struct  buf *dt;
	struct  buf *bp;

	dt = d->d_devtab;
	if (d->d_dir == FOREWARD && dt->b_actf == 0)
		d->d_dir = BACKWARD;
	else if (d->d_dir == BACKWARD && dt->b_actl == 0)
		d->d_dir = FOREWARD;
	if (d->d_dir == FOREWARD)
		if (bp = dt->b_actf) {
			dt->b_actf = bp->av_forw;
			d->d_blkno = bp->b_blkno;
		} else
			ipanic("disk: getnxtbp but dt->b_actf == 0");
	else
		if (bp = dt->b_actl) {
			dt->b_actl = bp->av_forw;
			d->d_blkno = bp->b_blkno;
		} else
			ipanic("disk: getnxtbp but dt->b_actl == 0");
	return(bp);
}

/*
 * Make a channel program for the block bp.
 */
static
makecprog(bp, d)
struct  buf *bp;
register struct  dasd *d;
{
	struct  dskdesc *dd;
        int     rec, tmp;

	if (bp->b_un.b_addr == 0)
		ipanic("disk: zero data address");
	if(bp->b_blkno < 0 || bp->b_blkno >= d->d_maxblk)
		ipanic("disk: bad block number");
	dd = d->d_desc;
        tmp = bp->b_blkno;
        rec = (tmp % dd->d_blktrk) + 1;
        tmp = (tmp / dd->d_blktrk) + TRKSKP;
        d->d_recloc[d->d_cpend].d_cc = tmp / dd->d_trkcyl;
        d->d_recloc[d->d_cpend].d_hh = tmp % dd->d_trkcyl;
	d->d_recloc[d->d_cpend].d_r0 = rec << 8;
        d->d_recloc[d->d_cpend].d_sector = (dd->d_const + (rec-1)*dd->d_factor) / dd->d_divsor; /* RPS */
        if (bp->b_flags & B_READ)
                d->d_ccws[d->d_cpend][4].cc_cmd = RDDATA;
        else {
		if(!d->d_rw)
			ipanic("disk: writing on R/O disk");
                d->d_ccws[d->d_cpend][4].cc_cmd = WRTDATA;
	}
        d->d_ccws[d->d_cpend][4].cc_addr = (int)bp->b_un.b_addr;
	d->d_ccws[d->d_cpend][4].cc_cc = 1;
}
 
/*
 * DASD interrupt routine
 */
dskintr(d, csw, sense)
register struct dasd *d;
csw_t *csw;
char *sense;
{
        int dskaint();
	struct buf *bp;
        struct buf *dt;
	struct recloc *rp;
	int cppc, ncp;
	int i, retry;
	int errdsp, restdsp, fwddsp;
 
        dt = d->d_devtab;
	if (csw->cs_dblw == 0L) {
		/*
		 * Missing interrupt.
		 */
		printf("dskintr - missing interrupt on %x\n", d->d_devaddr);
		sio(d->d_devaddr, &d->d_ccws[d->d_cpst][0], dskintr, (int)d);
		return;
	}

	/*
	 * First take care of all blocks on which io completed normally.
	 * The channel program program counter in the csw points 8 past
	 * the last channel command executed.
	 */
	if (csw->cs_word2 & (UC | DE | IL)) {
		/*
		 * Get the channel program program counter from the csw,
		 * check to make sure it is in range (if not, the channel
		 * program probably got nowhere), then compute the number
		 * of the channel program the interrupt is for. All
		 * channel programs before that must have terminated
		 * normally, so call iodone for the associated blocks.
		 */
		cppc = csw->cs_addr;
		if (cppc >= (int)d->d_ccws[d->d_cpst] && cppc <= (int)d->d_ccws[d->d_cpend])  {
			ncp = (cppc - (int)d->d_ccws[0] - 1) / (CPSIZE * sizeof(ccw_t));
			if (ncp > d->d_cpst)
			        dt->b_errcnt = 0;
			while (d->d_cpst < ncp) {
				bp = d->d_iolist->av_forw;
				iodone (d->d_iolist);
				d->d_iolist = bp;
				d->d_qlen--;
				d->d_cpst++;
			}
		}
	}
        bp = d->d_iolist;
	/*
	 * Now take care of the channel program for which the interrupt
	 * is being given.
	 */
        if (csw->cs_uc) {
		prdev("unit check", bp->b_dev);
		printf("sense=");
		for (i=0; i<24; i++)
			printf("%2x", sense[i]);
		printf("\n");
		rp = &d->d_recloc[d->d_cpst];
		printf("blockno %d bb %d cc %d hh %d r0>>8 %d sector %d\n",
			bp->b_blkno, rp->d_bb, rp->d_cc, rp->d_hh, rp->d_r0>>8, rp->d_sector);
		retry = 0;
		if (((BUSOUT || ENVDATA) && !PERMERR) || INTREQ) {
			if (dt->b_errcnt++ == 0)
			        retry = 1;
		} else if ((EQCHK || DATACHK || OVERRUN) && !PERMERR) {
			if (dt->b_errcnt++ < 10)
			        retry = 1;
		}
		if (retry) {
			sio(d->d_devaddr, &d->d_ccws[d->d_cpst][0], dskintr, (int)d);
			return;
		}
		if (INTREQ) {
                        d->d_intreq = 1;
                        printf("Intervention required on %3x\n", d->d_devaddr);
                        setax(d->d_devaddr, dskaint, (int)d);
                        return;
                }
		if (DATACHK && CRCTABL && ((sense[23] & 01) == 0)) {
			/*
			 * Apply error correction function defined in the manual.
			 * GA26-1617, p. 76.
			 */
			errdsp = (sense[18] << 8) | sense[19];
			restdsp = (sense[15] << 16) | (sense[16] << 8) | sense[17];
			fwddsp = restdsp - errdsp;
                        printf("Error correction applied on dev %d block %d ",
			        bp->b_dev, bp->b_blkno);
                        printf("errdsp %d restdsp %d fwddsp %d\n", errdsp, restdsp, fwddsp);
			if (errdsp > 3)
				errdsp = 3;
			if (fwddsp >= 0 && fwddsp + errdsp <= BSIZE)
			        for (i = 0; i < errdsp; i++)
				        ((char *)bp->b_un.b_addr)[fwddsp+i] ^= sense[20+i];
			else
				ipanic("Disk: error correction out of bounds");
		} else {
	                bp->b_flags |= B_ERROR;
		}
	}
	if(csw->cs_il) {
		bp->b_flags |= B_ERROR;
	}
        if(csw->cs_uc || csw->cs_il) {
		dt->b_errcnt = 0;
                d->d_qlen--;
		d->d_iolist = bp->av_forw;
                iodone(bp);
		if (++d->d_cpst < d->d_cpend) {
			/*
			 * If the error occurred in the middle of a string
			 * of channel programs, then the rest of the
			 * programs must be started now.
			 * It should be noted that this is the only reason
			 * for the existence of d->d_cpst.
			 */
			sio(d->d_devaddr, &d->d_ccws[d->d_cpst][0], dskintr, (int)d);
			return;
		}
		dt->b_active = 0;
                if (dt->b_actf || dt->b_actl)
                        dskstart(d);
		else
			wakeup((caddr_t)dt);
                return;
        }
        if (csw->cs_de) {
                if (dt->b_active != 1)
                        ipanic("disk: unexpected DE");
		if (cppc != (int)&d->d_ccws[d->d_cpend][0])
			ipanic("disk: DE but channel program not done");
		dt->b_errcnt = 0;
                dt->b_active = 0;
                d->d_qlen--;
                iodone(bp);
                if (dt->b_actf || dt->b_actl)
                        dskstart(d);
		else
			wakeup((caddr_t)dt);
        }
	if (csw->cs_cc == 3) {            /* device not working */
	        dt->b_active = 0;
		d->d_dead = 1;
		while (d->d_cpst < d->d_cpend) {
			bp = d->d_iolist->av_forw;
			d->d_iolist->b_flags |= B_ERROR;
			iodone(d->d_iolist);
			d->d_iolist = bp;
			d->d_qlen--;
			d->d_cpst++;
		}
	}
	if (csw->cs_cc != 3 && (csw->cs_word2 & (UC | DE)) == 0)
		printf("Unknown csw on dev %x %16lx\n",
			d->d_devaddr, csw->cs_dblw);
}
 
/*
 * Asynchronous DASD interrupt.
 * Used only when intervention is required on the device.
 * ARGSUSED
 */
dskaint(d, csw, sense)
register struct dasd *d;
csw_t *csw;
char *sense;
{
        if (d->d_intreq && csw->cs_de) {
                d->d_intreq = 0;
                sio(d->d_devaddr, &d->d_ccws[d->d_cpst][0], dskintr, (int)d);
        }
}
 
/*
 * Determine DASD type.
 * This is done by seeking to the highest track possible
 * for a type and seeing if it works.
 */
struct dskdesc *
dsktype(d)
register struct dasd *d;
{
        struct dskdesc *dd;
        int i, dskiint();
 
        d->d_recloc[0].d_bb = 0;
        d->d_recloc[0].d_cc = 0;
	d->d_ccws[0][0].cc_dblw = 0;
        d->d_ccws[0][0].cc_cmd = SEEK;
        d->d_ccws[0][0].cc_addr = (int)&d->d_recloc[0].d_bb;
        d->d_ccws[0][0].cc_count = 6;
        for (i=0; i < NTYPS; i++) {
                d->d_recloc[0].d_hh = dskdesc[i].d_trkcyl - 1;
                d->d_done = 0;
                sio(d->d_devaddr, &d->d_ccws[0][0], dskiint, (int)d);
                while (d->d_done == 0)
			sleep((caddr_t)&d->d_done, DSKPRI);
                switch (d->d_done) {
                case 1:
                        dd = &dskdesc[i];
                        goto out;
                case 2:
                        break;
                case 3:
                        dd = 0;
                        goto out;
                }
        }
        dd = 0;
out:
        return(dd);
}
 
/*
 * Determine the number of cylinders on a DASD.
 * This is done using a binary search of seeks.
 */
dskcyls(d)
register struct dasd *d;
{
        int dskiint();
        register maxgood, minbad;
 
        maxgood = 0;
        minbad = MAXCYL;
	d->d_ccws[0][0].cc_dblw = 0;
        d->d_ccws[0][0].cc_cmd = SEEK;
        d->d_ccws[0][0].cc_addr = (int) &d->d_recloc[0].d_bb;
        d->d_ccws[0][0].cc_count = 6;
        d->d_recloc[0].d_bb = 0;
        d->d_recloc[0].d_hh = 0;
        while (maxgood < minbad - 1) {
                d->d_recloc[0].d_cc = (maxgood+minbad) / 2;
                d->d_done = 0;
                sio(d->d_devaddr, &d->d_ccws[0][0], dskiint, (int)d);
                while (d->d_done == 0)
			sleep((caddr_t)&d->d_done, DSKPRI);
                if (d->d_done == 1)
			maxgood = d->d_recloc[0].d_cc;
                else
			minbad = d->d_recloc[0].d_cc;
        }
        return(maxgood+1);
}
 
/*
 * DASD initialization interrupt.
 */
dskiint(d, csw, sense)
register struct dasd *d;
csw_t *csw;
char *sense;
{
        if (csw->cs_uc) {
		if(CMDREJ)
                        d->d_done = 2;
		else
                        d->d_done = 3;
        } else if (csw->cs_de)
                d->d_done = 1;
        if (csw->cs_cc == 3)
                d->d_done = 3;
        if (d->d_done)
                wakeup((caddr_t)&d->d_done);
}
 
/*
 * Check results of tio.
 * ARGSUSED
 */
dsktioi(d, csw, sense)
register struct dasd *d;
csw_t *csw;
char *sense;
{
        if (csw->cs_cc == 3)
                d->d_done = 3;
        else if (csw->cs_uc)
                d->d_done = 2;
        else
                d->d_done = 1;
        wakeup((caddr_t)&d->d_done);
}
