/*
 * Maintain and manipulate an in-core image of
 * what the screen looks like at any time
 */

#include <tcl/3270.h>
#include <tcl/screen.h>
#include <tcl/tclio.h>
#include <stdio.h>

/*
 * Update the screen image from 3270 input
 */
scrinput()
{
	register char *bp;
	register int i;
	char aid;

	aid = inbuf.data[0];
	if(aid == SOH)
		aid = TESTREQ;
	if(aid == CLEAR) {
		clear(scr_data, SCRSIZE);
		clear(scr_attr, SCRSIZE);
		bufaddr = 0;
		cursaddr = 0;
		return;
	}
	if(inbuf.op == READBUF) {
		bp = &inbuf.data[1];
		cursaddr = ((bp[0] & 077) << 6) | (bp[1] & 077);
		bp += 2;
		curattr = 0;
		for(i=0; i < SCRSIZE; i++, bp++) {
			if(*bp == SF) {
				scr_data[i] = 0;
				curattr = *++bp & 077;
				scr_attr[i] = curattr | STARTFLD;
			} else {
				scr_data[i] = tubitab[*bp];
				scr_attr[i] = curattr;
			}
		}
		for(i=0; (scr_attr[i] & STARTFLD) == 0 && i < SCRSIZE; i++)
			scr_attr[i] = curattr;
		return;
	}
	if(aid == PA1 || aid == PA2)
		return;
	if(aid == TESTREQ) {
		bp = &inbuf.data[4];
	} else {
		bp = &inbuf.data[1];
		cursaddr = ((bp[0] & 077) << 6) | (bp[1] & 077);
		bp += 2;
	}
	while(bp < &inbuf.data[inbuf.len]) {
		if(*bp == SBA) {
			bufaddr = ((bp[1] & 077) << 6) | (bp[2] & 077);
			bp += 3;
			curattr = scr_attr[bufaddr];
			i = bufaddr;
			decmod(i, SCRSIZE);
			scr_attr[i] |= MDT;
		} else {
			scr_data[bufaddr] = tubitab[*bp++];
			scr_attr[bufaddr] = curattr;
			incmod(bufaddr, SCRSIZE);
		}
	}
}

/*
 * Update the screen image from 3270 output
 */
scroutput()
{
	char *bp, c;
	int i, j, newaddr, lastcmd;

	if(outbuf.op == EWRITE) {
		clear(scr_data, SCRSIZE);
		clear(scr_attr, SCRSIZE);
		bufaddr = 0;
		cursaddr = 0;
	}
	if(outbuf.data[0] & RSTMDT) {
		for(i=0; i<SCRSIZE; i++)
			if(scr_attr[i] & MDT)
				scr_attr[i] &= ~MDT;
	}
	bp = &outbuf.data[1];
	lastcmd = 1;
	while(bp < &outbuf.data[outbuf.len]) switch(*bp) {

	case SF:        /* start field */
		curattr = bp[1] & 077;
		scr_attr[bufaddr] = STARTFLD | curattr;
		scr_data[bufaddr] = 0;
		incmod(bufaddr, SCRSIZE);
		for(i=bufaddr; (scr_attr[i] & STARTFLD) == 0; incmod(i, SCRSIZE))
                        scr_attr[i] = curattr;
		bp += 2;
		lastcmd = SF;
		break;

	case SBA:       /* set buffer addr */
		bufaddr = ((bp[1] & 077) << 6) | (bp[2] & 077);
		if(bufaddr >= SCRSIZE) {
			fprintf(stderr, "bad buffer address\n");
			bufaddr = 0;
			return;
		}
		curattr = scr_attr[bufaddr] & ~STARTFLD;
		bp += 3;
		lastcmd = SBA;
		break;

	case IC:        /* insert cursor */
		cursaddr = bufaddr;
		bp += 1;
		lastcmd = IC;
		break;

	case RA:        /* repeat-to-address */
		newaddr = ((bp[1] & 077) << 6) | (bp[2] & 077);
		c = tubitab[bp[3]];
		for(i=0; bufaddr!=newaddr || i==0; i++) {
			scr_data[bufaddr] = c;
			scr_attr[bufaddr] = curattr;
			incmod(bufaddr, SCRSIZE);
		}
		for(i=bufaddr, j=0; (scr_attr[i] & STARTFLD)==0 && j<SCRSIZE ; incmod(i, SCRSIZE), j++)
                        scr_attr[i] = curattr;
		bp += 4;
		lastcmd = RA;
		break;

	case EUA:       /* erase unprotected to address */
		newaddr = ((bp[1] & 077) << 6) | (bp[2] & 077);
		for(i=0; bufaddr!=newaddr || i==0; i++) {
			if((scr_attr[bufaddr] & PROT) == 0)
				scr_data[bufaddr] = 0;
			incmod(bufaddr, SCRSIZE);
		}
		curattr = scr_attr[bufaddr];
		bp += 3;
		lastcmd = EUA;
		break;

	case PT:        /* program tab */
                while((scr_attr[bufaddr] & (STARTFLD+PROT)) != STARTFLD
                     && bufaddr != SCRSIZE-1) {
			if(scr_attr[bufaddr] & STARTFLD)
				lastcmd = PT;
			if(lastcmd == 0)
				scr_data[bufaddr] = 0;
			incmod(bufaddr, SCRSIZE);
		}
                incmod(bufaddr, SCRSIZE);
		curattr = scr_attr[bufaddr];
		bp += 1;
		break;

	default:        /* a real character */
		scr_data[bufaddr] = tubitab[*bp++];
		scr_attr[bufaddr] = curattr;
		incmod(bufaddr, SCRSIZE);
		lastcmd = 0;
		break;
	}
}

/*
 * Dump the current screen image into the outbuf buffer
 * Used to restore and generate screens
 */
outscreen()
{
	register char *p;
	register int i;

	outbuf.op = EWRITE;
	p = outbuf.data;
	*p++ = lastwcc;
	sba(p, 0);
	for(i=0; i<SCRSIZE; i++) {
		if(scr_attr[i] & STARTFLD) {
			sf(p, scr_attr[i] & ~STARTFLD);
		} else {
			*p++ = tubotab[scr_data[i]];
		}
	}
	sba(p, cursaddr);
	ic(p);
	outbuf.len = p - outbuf.data;
}


/*
 * Write the current screen image to a file
 */
wrtscreen(app, fn)
char *fn;
{
	static FILE *fp = NULL;
	int r, c;
	char xrow[81], *rp, *sp;

	if(fn == NULL)
		fp = stdout;
	else if(fn[0] != '\0') {
		if(fp != NULL)
			fclose(fp);
		if(app)
			fp = fopen(fn, "a");
		else
                        fp = fopen(fn, "w");
	}
	if(fp == NULL)
		return;
	for(r=1; r<=24; r++) {
		rp = &xrow[80];
		*rp-- = 0;
		sp = screen(rc(r,80));
		for(c=79; c>=0; c--) {
			if(*sp == 0 || *sp == ' ') {
				*rp-- = 0;
				sp--;
			} else {
				*rp-- = *sp--;
				break;
			}
		}
		for(; c>=0; c--) {
			if(*sp == 0 || *sp == ' ')
				*rp-- = ' ';
			else
				*rp-- = *sp;
			sp--;
		}
		fprintf(fp, "%s\n", xrow);
	}
	fflush(fp);
}

/*
 * Read a file and put it into the current screen image
 */
rdscreen(fld, fn)
char *fn;
{
	static FILE *fp = NULL;
	char line[81];

	if(fn[0] != '\0') {
		if(fp != NULL)
                        fclose(fp);
		fp = fopen(fn, "r");
	}
	if(fp == NULL)
		return;
	cursaddr = fld;
	while(cursaddr >= fld) {
		if(fgets(line, 81, fp) == NULL) break;
		inputtxt(line);
	}
}

clear(cp, n)
register char *cp;
register int n;
{
	while(n--) *cp++ = 0;
}
