/*
 * Allow 3270 and LDEV to interact
 * Signals are used to notify the process
 * when data arrives from the 3270 while
 * reading from LDEV, or vice-versa
 */
#include <tcl/tclio.h>
#include <sgtty.h>
#include <signal.h>
#include <setjmp.h>

#define POKESIG SIGMC

static jmp_buf polljmp;

/*
 * Catch the POKE signal and bounce back up to the main loop
 */
static catch()
{
	signal(POKESIG, catch);
	longjmp(polljmp);
}

/*
 * Run interaction:
 * 1. If there is LDEV data, read it and write it to 3270
 * 2. If there is 3270 data, or an unsolicited read has been
 *    requested, read 3270 data and write to LDEV
 * 3. If the 3270 keyboard is locked, read from LDEV but allow
 *    signal if 3270 input occurs
 * 4. Otherwise, read from 3270 but allow signal if LDEV data appears
 */
interact()
{
	int st = 0;

	if(ldfd <= 0)
		return;
	if(fsfd <= 0)
		fsopen();
	if(outbuf.op != 0) {
		outscreen();
		putoutput();
	}

	signal(POKESIG, catch);
	setjmp(polljmp);
	while(st < 2) {
		if(poll(ldfd)) {
			if((st = getoutput()) == 0)
                                putoutput();
			continue;
		}
		if(unsolicited || poll(fsfd)) {
			if((st = getinput()) == 0)
                                putinput();
			continue;
		}
		if(kbd == LOCKED) {
			setpoke(fsfd);
			st = getoutput();
			rstpoke(fsfd);
			if(st == 0)
                                putoutput();
			continue;
		}
		setpoke(ldfd);
		st = getinput();
		rstpoke(ldfd);
		if(st == 0)
                        putinput();
	}
}

/*
 * Test to see if data is ready to be read
 */
poll(fd)
{
	int x;

	if(ioctl(fd, TIOPOLL, &x))
		return(0);
	return(x);
}

/*
 * Set up to get a signal when data appears
 */
static setpoke(fd)
{
	ioctl(fd, TIOPOKE, POKESIG);
}

/*
 * Undo setpoke
 */
static rstpoke(fd)
{
	ioctl(fd, TIONPOKE, 0);
}
