/*
 * timec - convert from string specification of time to UNIX integer time
 *
 * Usage:
 *      t = timec(str, tense);
 *      char *str;
 *      int tense, timec();
 *
 * 'str' points to  a null-terminated ASCII string which contains the time
 * specification to be interpreted. 'tense' sets the default tense
 * of the specified time: -1 is past tense, 0 is present, 1 is future.
 * For invalid time specifications, -1 is returned.
 *
 * Many, many things are valid time specifications, a few examples follow:
 *      11/22/79 3:23 PM
 *      November 22, 1979 at 3:23 in the afternoon
 *      15:23 on nov 11 79
 *      now
 *      3 hours from now
 *      ten years ago
 *      yesterday
 *      next friday
 *      last december
 *      two o'clock
 *
 * The tense parameter specifies how some times are to be interpreted.
 * If the tense is -1, the time is expected to refer to a time in the past.
 * If the tense is 1, the time is expected to be in the future..
 * If the tense is 0, the time will be the 'nearest' time that matches.
 *
 * Examples:   Time Now    Str   Tense   Result
 *              2:00 PM   3:00     0     3:00 PM
 *              2:00 PM   3:00    -1     3:00 AM
 *              2:00 PM   3:00     1     3:00 PM
 *              4:00 PM   3:00     0     3:00 PM
 *              4:00 PM   3:00    -1     3:00 PM
 *              4:00 PM   3:00     1     3:00 AM next day
 *             April '80  June     0     June '80
 *             April '80  June    -1     June '79
 *             April '80  June     1     June '80
 *              July '80  June     0     June '80
 *              July '80  June    -1     June '80
 *              July '80  June     1     June '81
 *
 * Note that specifying past or future tense does not gaurantee
 * that the time returned will be in the past or future.
 * This occurs when 'str' contains explicit info such as 2/2/80.
 */
%token BEFORE AFTER AGO HENCE
%token NUMBER UNIT THIS NEXT LAST
%token MONTH AM PM NOW TODAY
%token SLASHD COLONT TOMORROW YESTERDAY
%token WEEKDAY DAYNUM YEARNUM HOURNUM
%token EPOCH
%{
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/timeb.h>
static int date(), index(), isdst();
static char *token();
static int number, slashd, colont, month, today, hday, unit;
static int unitval[] = { 1, 60, 60*60, 24*60*60, 7*24*60*60, 30*24*60*60, 365*24*60*60 };
static struct tm *have;
static int now, weekday;
static char *timestr;
static int req_tense;
static int tcval, tcerr;
static int abstime;
%}
%%
start:  time                        = { tcval = $1; }
        | error                     = { tcerr = 1; }
time:   abstime                     = { $$ = $1; }
	| reltime BEFORE time       = { $$ = $3 - $1; }
	| reltime AFTER time        = { $$ = $3 + $1; }
	| reltime AGO               = { $$ = now - $1; }
	| reltime HENCE             = { $$ = now + $1; }
reltime:  factor UNIT               = { $$ = $1 * unitval[unit]; }
abstime:  THIS UNIT                 = { $$ = tnl(unit, 0); }
        | NEXT UNIT                 = { $$ = tnl(unit, 1); }
        | LAST UNIT                 = { $$ = tnl(unit, -1); }
        | WEEKDAY MONTH day COLONT year = {
				/* inverse of ctime */
				$$ = date(month,$3,$5) + colont - hday;
			}
        | date
        | htime
        | date htime                = { $$ = $1 + $2 - hday; }
        | htime date                = { $$ = $2 + $1 - hday; }
        | NOW                       = { $$ = now; }
        | EPOCH                     = { $$ = 0; }
date:     MONTH day                 = { $$ = date(month, $2, -1); }
        | MONTH                     = { $$ = date(month, 1, -1); }
        | MONTH day year            = { $$ = date(month, $2, $3); }
        | MONTH year                = { $$ = date(month, 1, $2); }
        | TODAY                     = { $$ = today; }
        | TOMORROW                  = { $$ = today + 24*60*60; }
        | YESTERDAY                 = { $$ = today - 24*60*60; }
        | SLASHD                    = { $$ = slashd; }
        | WEEKDAY                   = { $$ = dow(weekday, req_tense); }
        | THIS WEEKDAY              = { $$ = dow(weekday, 0); }
        | NEXT WEEKDAY              = { $$ = dow(weekday, 1); }
        | LAST WEEKDAY              = { $$ = dow(weekday, -1); }
        | THIS MONTH                = { $$ = moy(month, 0); }
        | NEXT MONTH                = { $$ = moy(month, 1); }
        | LAST MONTH                = { $$ = moy(month, -1); }
htime:    COLONT                    = { $$ = colont; }
        | COLONT AM                 = { $$ = colont - hday + today; hday = today; }
        | COLONT PM                 = { $$ = colont - hday + today + 12*60*60; hday = today; }
        | AM                        = { $$ = hday; }
        | THIS AM                   = { $$ = hday; }
        | PM                        = { $$ = hday + 12*60*60; }
        | THIS PM                   = { $$ = hday + 12*60*60; }
        | hour                      = { $$ = htime($1, 0, 0); }
        | hour AM                   = { $$ = hday + ($1%12)*60*60; }
        | hour PM                   = { $$ = hday + (($1%12)+12)*60*60; }
hour:     HOURNUM                   = { $$ = number; }
day:      DAYNUM                    = { $$ = number; }
        | HOURNUM                   = { $$ = number; }
year:     YEARNUM                   = { $$ = number; }
factor:   NUMBER                    = { $$ = number; }
        | DAYNUM                    = { $$ = number; }
        | YEARNUM                   = { $$ = number; }
        | HOURNUM                   = { $$ = number; }
%%
/*
 * Main entry - just figure out the current time and call yyparse
 * On return, check for errors
 */
timec(tmstr, tense)
char *tmstr;
{
	int ndst, tdst;

	req_tense = tense;
	timestr = tmstr;
        now = time();
        have = localtime(now);
        today = now - (3600*have->tm_hour+60*have->tm_min+have->tm_sec);
	hday = today;
	tcerr = 0;
	abstime = 0;
	if(yyparse() || tcerr)
		return(-1);
	ndst = isdst(now);
	tdst = isdst(tcval);
	if(abstime) {
		if(tdst)
                        tcval -= 60*60;
	} else {
		if(ndst != tdst)
			if(ndst)
				tcval += 60*60;
			else
				tcval -= 60*60;
	}
	return(tcval);
}

/*
 * Error routine - just set error indication
 */
static yyerror(s)
char *s;
{
	tcerr = 1;
}

/*
 * Lexical analyzer:
 * Get a token and see if it's a slash date(2/23/80) or a colon time
 * (3:42:56) or a number or a word. If it's a noise word, repeat.
 */
static yylex()
{
	char *t, *token();
	int m,d,y,h,s,type;
loop:
	t = token();
	if(t == 0)
		return(0);
	if(index(t, '/')) {     /* slash date */
		m = 0;
		while(isdigit(*t))
			m = 10*m + *t++ - '0';
		if(*t++ != '/') yyerror("bad slash date");
		d = 0;
		while(isdigit(*t))
			d = 10*d + *t++ - '0';
		if(*t == '\0') {
			slashd = date(m, d, -1);
			return(SLASHD);
		}
		if(*t++ != '/') yyerror("bad slash date");
		y = 0;
		while(isdigit(*t))
			y = 10*y + *t++ - '0';
		slashd = date(m, d, y);
		return(SLASHD);
	}
	if(index(t, ':')) {     /* colon time */
		h = 0;
		while(isdigit(*t))
			h = 10*h + *t++ - '0';
		if(*t++ != ':') yyerror("bad colon time");
		m = 0;
		while(isdigit(*t))
			m = 10*m + *t++ - '0';
		if(h >= 24 || m >= 60)
			yyerror("bad colon time");
		if(*t == '\0') {
			colont = htime(h, m, 0);
			return(COLONT);
		}
		if(*t++ != ':') yyerror("bad colon time");
		s = 0;
		while(isdigit(*t))
			s = 10*s + *t++ - '0';
		if(s >= 60)
			yyerror("bad colon time");
		colont = htime(h, m, s);
		return(COLONT);
	}
	if(isdigit(*t)) {       /* number */
		number = atoi(t);
		if(number <= 12) return(HOURNUM);
		if(number <= 31) return(DAYNUM);
		if(number >= 70 && number <= 99) return(YEARNUM);
		if(number >= 1970 && number <= 1999) return(YEARNUM);
		return(NUMBER);
	}
	type = lookup(t);
	if(type == -1) goto loop;
	return(type);
}

/*
 * Get the next token from the input string
 */
static char *token()
{
	static char tok[50], eof;
	char *p;
	int c;

	if(eof) {
		eof = 0;
		return(0);
	}
	while((c = *timestr++) == ' ' || c == '\t' || c == '\n' ||
		c == ',');
	if(c == '\0') return(0);
	p = tok;
	*p++ = c;
	while((c = *timestr++) != ' ' && c != '\t' && c != '\n' &&
		c != ',' && c != '\0')
		*p++ =  c;
	if(c == '\0') eof = 1;
	*p = '\0';
	return(tok);
}


/*
 * Table of words and what they "mean".
 */
static struct word {
	char *name;
	int type;
	int value;
} words [] = {
	"the", -1, 0,
	"at", -1, 0,
	"in", -1, 0,
	"a", NUMBER, 1,
	"tomorrow", TOMORROW, 0,
	"yesterday", YESTERDAY, 0,
	"before", BEFORE, 0,
	"after", AFTER, 0,
	"ago", AGO, 0,
	"hence", HENCE, 0,
	"epoch", EPOCH, 0,
	"this", THIS, 0,
	"next", NEXT, 0,
	"last", LAST, 0,
	"now", NOW, 0,
	"today", TODAY, 0,
	"from", AFTER, 0,
	"second", UNIT, 0,
	"seconds", UNIT, 0,
	"minute", UNIT, 1,
	"minutes", UNIT, 1,
	"hour", UNIT, 2,
	"hours", UNIT, 2,
	"day", UNIT, 3,
	"days", UNIT, 3,
	"week", UNIT, 4,
	"weeks", UNIT, 4,
	"month", UNIT, 5,
	"months", UNIT, 5,
	"year", UNIT, 6,
	"years", UNIT, 6,
	"am", AM, 0,
	"a.m.", AM, 0,
	"morning", AM, 0,
	"pm", PM, 0,
	"p.m.", PM, 0,
	"afternoon", PM, 0,
	"january", MONTH, 1,
	"jan", MONTH, 1,
	"february", MONTH, 2,
	"feb", MONTH, 2,
	"march", MONTH, 3,
	"mar", MONTH, 3,
	"april", MONTH, 4,
	"apr", MONTH, 4,
	"may", MONTH, 5,
	"june", MONTH, 6,
	"jun", MONTH, 6,
	"july", MONTH, 7,
	"jul", MONTH, 7,
	"august", MONTH, 8,
	"aug", MONTH, 8,
	"september", MONTH, 9,
	"sep", MONTH, 9,
	"october", MONTH, 10,
	"oct", MONTH, 10,
	"november", MONTH, 11,
	"nov", MONTH, 11,
	"december", MONTH, 12,
	"dec", MONTH, 12,
	"sunday", WEEKDAY, 0,
	"sun", WEEKDAY, 0,
	"monday", WEEKDAY, 1,
	"mon", WEEKDAY, 1,
	"tuesday", WEEKDAY, 2,
	"tue", WEEKDAY, 2,
	"wednesday", WEEKDAY, 3,
	"wed", WEEKDAY, 3,
	"thursday", WEEKDAY, 4,
	"thu", WEEKDAY, 4,
	"friday", WEEKDAY, 5,
	"fri", WEEKDAY, 5,
	"saturday", WEEKDAY, 6,
	"sat", WEEKDAY, 6,
	"first", DAYNUM, 1,
	"second", DAYNUM, 2,
	"third", DAYNUM, 3,
	"fourth", DAYNUM, 4,
	"fifth", DAYNUM, 5,
	"sixth", DAYNUM, 6,
	"seventh", DAYNUM, 7,
	"eighth", DAYNUM, 8,
	"ninth", DAYNUM, 9,
	"tenth", DAYNUM, 10,
	"eleventh", DAYNUM, 11,
	"twelvth", DAYNUM, 12,
	"thirteenth", DAYNUM, 13,
	"fourteenth", DAYNUM, 14,
	"fifteenth", DAYNUM, 15,
	"sixteenth", DAYNUM, 16,
	"seventeenth", DAYNUM, 17,
	"eighteenth", DAYNUM, 18,
	"nineteenth", DAYNUM, 19,
	"twentieth", DAYNUM, 20,
	"twenty-first", DAYNUM, 21,
	"twenty-second", DAYNUM, 22,
	"twenty-third", DAYNUM, 23,
	"twenty-fourth", DAYNUM, 24,
	"twenty-fifth", DAYNUM, 25,
	"twenty-sixth", DAYNUM, 26,
	"twenty-seventh", DAYNUM, 27,
	"twenty-eighth", DAYNUM, 28,
	"twenty-ninth", DAYNUM, 29,
	"thirtieth", DAYNUM, 30,
	"thirty-first", DAYNUM, 31,
	"o'clock", -1, 0,
	"one", HOURNUM, 1,
	"two", HOURNUM, 2,
	"three", HOURNUM, 3,
	"four", HOURNUM, 4,
	"five", HOURNUM, 5,
	"six", HOURNUM, 6,
	"seven", HOURNUM, 7,
	"eight", HOURNUM, 8,
	"nine", HOURNUM, 9,
	"ten", HOURNUM, 10,
	"eleven", HOURNUM, 11,
	"twelve", HOURNUM, 12,
	"thirteen", HOURNUM, 13,
	"fourteen", HOURNUM, 14,
	"fifteen", HOURNUM, 15,
	"sixteen", HOURNUM, 16,
	"seventeen", HOURNUM, 17,
	"eighteen", HOURNUM, 18,
	"nineteen", HOURNUM, 19,
	"twenty", HOURNUM, 20,
	"twenty-one", HOURNUM, 21,
	"twenty-two", HOURNUM, 22,
	"twenty-three", HOURNUM, 23,
	"twenty-four", HOURNUM, 24,
	0,
};

/*
 * Look up a word by linear search (slow).
 */
static lookup(t)
char *t;
{
	char *s;
	struct word *w, *x;

	for(s = t; *s; s++) if(isupper(*s)) *s = tolower(*s);
	x = NULL;
	for(w = words; w->name; w++)
	if(strcmp(t, w->name) == 0) {
		x = w;
		break;
	}
	if(x == NULL) {
		yyerror("bad word");
		return(0);
	}
	switch(x->type) {
	case MONTH:
		month = x->value;
		break;
	case UNIT:
		unit = x->value;
		break;
	case WEEKDAY:
		weekday = x->value;
		break;
	case NUMBER: case DAYNUM: case HOURNUM: case YEARNUM:
		number = x->value;
		break;
	}
	return(x->type);
}

/*
 * See if the character c is in the string s.
 */
static index(s, c)
char *s;
char c;
{
	while(*s) if(*s++ == c) return(1);
	return(0);
}

static int montab[] = { 31, 0, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

/*
 * Handle the words 'this', 'next', and 'last' when applied to time units.
 */
static tnl(unit, tense)
{
	int t, m, y;

	switch(unit) {
	case 0: /* second */
		return(now + tense);
	case 1: /* minute */
		t = now;
		t -= have->tm_sec;
		return(t + tense*60);
	case 2: /* hour */
		t = now;
		t -= have->tm_sec;
		t -= 60 * have->tm_min;
		return(t + tense*60*60);
	case 3: /* day */
		return(today + tense*24*60*60);
	case 4: /* week */
		t = today;
		t -= have->tm_wday * 24*60*60;
		return(t + tense*7*24*60*60);
	case 5: /* month */
		m = 1 + have->tm_mon;
		y = have->tm_year;
		m += tense;
		if(m > 12) {
		        y++;
			m -= 12;
		}
		if(m <= 0) {
			y--;
			m += 12;
		}
		return(date(m, 1, y));
	case 6: /* year */
		return(date(1, 1, have->tm_year + tense));
	}
}

/*
 * Generate time given month, day, and possibly year.
 */
static date(m, d, y)
{
	int t, i;
	struct timeb timeb;

	if(y == -1) {
		switch(req_tense) {
		case 0: /* present */
			y = have->tm_year;
			break;
		case -1: /* past */
			if(m <= have->tm_mon+1)
				y = have->tm_year;
			else
				y = have->tm_year - 1;
			break;
		case 1: /* future */
			if(m > have->tm_mon+1)
				y = have->tm_year;
			else
				y = have->tm_year + 1;
			break;
		}
	}
	if(y >= 1970 && y < 2000)
		y -= 1900;
	else if(y < 70 || y > 99)
		yyerror("bad year");
	if(m < 1 || m > 12)
		yyerror("bad month");
	t = 0;
	for(i = 70; i < y; i++) {
		t += 365;
		if((i % 4) == 0) t += 1;
	}
	if((y % 4) == 0)
		montab[1] = 29;
	else
		montab[1] = 28;
	for(i = 1; i < m; i++) {
		t += montab[i-1];
	}
	if(d > montab[m-1])
		yyerror("bad day");
	t += d-1;
	ftime(&timeb);
	t *= 24*60*60;
	t += timeb.timezone*60;
	abstime = 1;
	return(t);
}

/*
 * Generate time given hour, minute, and second.
 */
static htime(h, m, s)
{
	int th, tw;

	switch(req_tense) {
	case 0: /* present */
		if(h < 12 && have->tm_hour >= 12)
			hday += 12*60*60;
		break;

	case -1: /* past */
		tw = 3600*h + 60*m + s;
		if(h > 12) { /* 24 hour time */
                        th = 3600*have->tm_hour + 60*have->tm_min + have->tm_sec;
			if(tw >= th)
				hday -= 24*60*60;
			break;
		}
		th = 3600*(have->tm_hour%12) + 60*have->tm_min + have->tm_sec;
		if(tw < th) {
			if(have->tm_hour >= 12)
				hday += 12*60*60;
		} else {
			if(have->tm_hour < 12)
				hday -= 12*60*60;
		}
		break;

	case 1: /* future */
		tw = 3600*h + 60*m + s;
		if(h > 12) { /* 24 hour time */
                        th = 3600*have->tm_hour + 60*have->tm_min + have->tm_sec;
			if(tw <= th)
				hday += 24*60*60;
			break;
		}
		th = 3600*(have->tm_hour%12) + 60*have->tm_min + have->tm_sec;
		if(tw > th) {
			if(have->tm_hour >= 12)
				hday += 12*60*60;
		} else {
			if(have->tm_hour < 12)
				hday += 12*60*60;
			else
				hday += 24*60*60;
		}
		break;
	}
	return(hday + 3600*h + 60*m + s);
}

/*
 * Generate time given day-of-week and tense.
 */
static dow(w, tense)
{
	int h, t;

	h = have->tm_wday;
	if(tense > 0 && w <= h) w += 7;
	if(tense < 0 && w >= h) w -= 7;
	t = today + (w-h) * 24*60*60;
	return(t);
}

/*
 * Do this, next, or last month_name
 */
static moy(m, tense)
{
	int y;

	switch(tense) {
	case 0: /* present */
		y = have->tm_year;
		break;
	case 1: /* future */
		if(m <= have->tm_mon+1)
			y = have->tm_year+1;
		else
			y = have->tm_year;
		break;
	case -1: /* past */
		if(m >= have->tm_mon+1)
			y = have->tm_year-1;
		else
			y = have->tm_year;
		break;
	}
	return(date(m, 1, y));
}

#define dysize(y) ((y%4) == 0 ? 366 : 365)

/*
 * Determine if tim is in daylight savings or not
 * This code is stolen and compressed from ctime(2)
 */
static isdst(tim)
{
	register int day, y, d;
	int hour, wday, daylb, dayle;
	struct timeb timeb;

	ftime(&timeb);
	if(timeb.dstflag == 0)
		return(0);
	tim -= timeb.timezone*60;
	hour = tim / 3600;
	day = hour / 24;
	hour = hour % 24;
	wday = (day+7340036) % 7;
	for(y=70; day >= dysize(y); y++)
		day -= dysize(y);
	d = 119 + dysize(y) - 365;
	daylb = d - (d - day + wday + 700) % 7; /* last sun in apr */
	d = 303 + dysize(y) - 365;
	dayle = d - (d - day + wday + 700) % 7; /* last sun in oct */
	if((day > daylb || (day == daylb && hour > 2)) &&
	   (day < dayle || (day == dayle && hour <= 1)))
		return(1);
	return(0);
}
