/*
 * p/n:
 * Program name:            Dump Formatter command execution
 * Author:                  Alan Ballard
 * Backup:
 * Description:
 *     This component contains the routine to execute dump formatter
 *     commands.
 */

#include "dump.h"
#include "tnode.h"
#include "y.tab.h"
#include <stdio.h>

extern FILE *out;

/*
 * dodump is the tree evaluation routine called from doreq for the
 * dump formatter commands.
 *
 * parameter savflag is set if the caller does not want his tree
 * discarded; otherwise the nodes are released as the routine
 * returns.
 */
dodump(p, savflag)
struct tnode *p;
int savflag;
{
	register struct tnode *t1, *t2;
        int     r, plen, temp;
	char    *buf;

	t1 = p->t_left;
	t2 = p->t_right;
	r = 0;
	switch(p->t_op) {
                /*
                 *  first, the commands...
                 */
                case FORM:
                        /*
                         *  Set special variables used to tell formatting
                         *  routines the device addresses.
                         */
                        io_chan = io_cu = io_unit = -1;

                        printformat(-1, nonecode, t1, savflag);
                        break;

                case PRINT:
                        printdump(p, savflag);
                        break;

                case MEM:
                case ADDR:
                        /*
                         * MEM and ADDR start out the same.
                         */
                        r = doreq(t1, savflag);
                        if (r < 0) {
                                error(0, "invalid expression");
                                break;
                        }
                        fprintf(out,"%06x = ", r);
                        symf(r);
                        fprintf(out,"\n");

                        if (p->t_op == ADDR) break;

                        /*
                         *  For MEM, get the memory and dump it too.
                         */
                        if (t2 == 0)
                                plen = 16;
                        else
                                plen = (int) t2;
                        while (plen > 0) {
                                temp = (plen > 4096) ? 4096 : plen;
                                buf = (char *) getblk(r, temp);
                                if (buf < 0)
                                        break;
                                pdump(r, buf, temp, 16);
                                plen -= temp;
                        }
                        break;

		default:
                        error(1, "unknown request %x", p->t_op);
        }
}

/*
 *  printdump performs the ;dump command by calling findstruct
 *  to locate and print various tables.
 */
printdump(expr, savflag)
struct tnode    *expr;
int             savflag;
{
        /*
         *  Set up to print out the process structures...
         */
        fprintf(out,"\n\n\nProcess, User and Tty Structures \n\n\n");
        findstruct(-1, nonecode, proccode, -1, expr, savflag);

        /*
         *  Now, the physical I/O structures
         */
        fprintf(out,"\n\n\nHardware I/O Structures \n\n\n");
        findstruct(-1, nonecode, chancode, -1, expr, savflag);

        /*
         *  Other I/O related ones
         */
        fprintf(out,"\n\n\nTube Structures \n\n\n");
        findstruct(-1, nonecode, tubecode, -1, (struct tnode *) NULL, savflag);

        /*
         *  File system related stuff.
         */
        fprintf(out,"\n\n\nFile System Structures \n\n\n");
        findstruct(-1, nonecode, dasdcode, -1, (struct tnode *) NULL, savflag);
        findstruct(-1, nonecode, filecode, -1, (struct tnode *) NULL, savflag);
        findstruct(-1, nonecode, mountcode, -1, (struct tnode *) NULL, savflag);
        findstruct(-1, nonecode, inodecode, -1, (struct tnode *) NULL, savflag);
        findstruct(-1, nonecode, bufcode, -1, (struct tnode *) NULL, savflag);

        /*
         *  Memory allocation state
         */
        fprintf(out,"\n\n\nMemory Allocation State \n\n\n");
        findstruct(-1, nonecode, coretabcode, -1, (struct tnode *) NULL, savflag);
}

/*
 *  printformat is the central recursive routine for printing specific
 *  structures and groups of structures.
 *
 *  It is called with the address of a structure, and an expression
 *  describing what it is supposed to do with it.  A null expression
 *  means print it; non-null means print associated structures described
 *  by the expression.
 */
printformat(addr, code, expr, savflag)
int     addr;           /* address of a structure */
enum structcode code;   /* code for what type of structure */
struct tnode    *expr;  /* expression for what to do */
int     savflag;        /* whether to free expression nodes */
{
        /*
         *  If called with an address of zero, just ignore it.
         *  This is used for null structure pointers most places.
         */
        if (addr == 0)
                return;
        if (expr == NULL) {
                /*
                 *  Print the structure we've already found.
                 */
                prt(addr, code);
                return;
         }

         /*
          *  Interpret expr
          */
         switch (expr->t_op) {

         case ADDREXPR:
                /*
                 * This isn't allowed if we already have anything.
                 */
                if (code != nonecode) {
                        error(0, "illegal use of address expression");
                        return;
                }
                addr = doreq(expr->t_left, savflag);
                /*
                 *  Now recurse, treating this address as being of the
                 *  required type.
                 */
                printformat(addr, expr->t_val, expr->t_right,
                        savflag);
                break;

         case TABLE:
                /*
                 *  A "TABLE" is unqualified structure name.
                 *
                 *  If the current request matches what we've already got,
                 *  peel it off and recurse.
                 *  This is to allow some harmless redundant requests.
                 */
                if (expr->t_val == code)
                    printformat(addr, code, expr->t_right, savflag);
                else
                    findstruct(addr, code, expr->t_val, -1, expr,
                             savflag);
                break;

         case ARR:
                findstruct(addr, code, expr->t_val, expr->t_left,
                        expr, savflag);
                break;

         case PROC:
                do_proc(addr, code, expr, savflag);
                break;

         case DEV:
                do_dev(addr, code, expr, savflag);
                break;

         case PRINT:
                do_dump(addr,code, expr, savflag);
                break;

         default:
                error(0, "invalid expr in printformat: op = %x",
                     expr->t_op);
         }
}

/*
 * do_dump is used during full dump processing to interleave the
 * formatting of some of the structures.
 */
do_dump(addr, code, expr, savflag)
int     addr;           /* address of a structure */
enum structcode code;   /* code for what type of structure we have */
struct tnode    *expr;  /* expression for what to do */
int     savflag;        /* whether to free expression nodes */
{
       switch (code) {
       case chancode:
               printformat(addr, code, (struct tnode *) NULL, savflag);
               findstruct(addr, code, ioqcode, -1, expr, 1);
               findstruct(addr, code, cucode, -1, expr, savflag);
               break;

       case cucode:
               printformat(addr,code, (struct tnode *) NULL, savflag);
               findstruct(addr, code, ioqcode, -1, (struct tnode *) NULL, 1);
               findstruct(addr, code, unitcode, -1, expr, savflag);
               break;

       case unitcode:
               printformat(addr, code, (struct tnode *) NULL, savflag);
               findstruct(addr, code, ioqcode, -1, (struct tnode *) NULL, 1);
               findstruct(addr, code, activecode, -1, (struct tnode *) NULL, savflag);
               break;

       case proccode:
               printformat(addr, code, (struct tnode *) NULL, savflag);
               findstruct(addr, code, usercode, -1, expr, savflag);
               break;

       case usercode:
               printformat(addr,code, (struct tnode *) NULL, savflag);
               findstruct(addr, code, ttycode, -1, (struct tnode *) NULL, savflag);
               findstruct(addr, code, stackcode, -1, (struct tnode *) NULL, savflag);
               break;

       }

}
