#include <stdio.h>
#include <ctype.h>

#define LINESIZE 512
#define EXPRSIZE  40

#define ALARM       0
#define BELL        1
#define CURSOR      2
#define ERASE       3
#define INIT        4
#define LOCK        5
#define NOBELL      6
#define NOERASE     7
#define NOINIT      8
#define NOLOCK      9
#define NOREAD     10
#define NORESETMDT 11
#define NOWAIT     12
#define NOWRITE    13
#define READ       14
#define RESETMDT   15
#define SLEEP      16
#define WAIT       17
#define WRITE      18

/*
 * When the generated code in the .c file is compiled the error
 * messages should refer to the .q file and have the
 * correct line numbers. To do this we use the #line feature of
 * the C preprocessor.
 * The generated sections of code are surrounded by { and }
 * so that they will act as a block in the rest of the code.
 * The two macros below do all of this.
 */

#define BEGIN_QS_LINES  output("#line %d \"%s\"\n", cline+2, cname);\
                        output("{\n");
#define   END_QS_LINES  output("}\n");\
                        output("#line %d \"%s\"\n", qline+1, qname);

int errors;             /* # of errors for current .q file */
int totalerrors;        /* total # of errors for all .q files */
char qname[30];         /* name of .q file */
char cname[30];         /* name of .c file */
int qline, cline;       /* line count in .c and .q files */
int escnl;              /* was a newline escaped? */
FILE *dotq, *dotc;      /* file pointers for .q and .c files */

char line[LINESIZE];    /* holds line read by input */
char *lp;               /* points to line */
char symbol[100];       /* holds symbol read by nextsym */
int skips;              /* number of blank lines skipped in panel */
int begincol;           /* beginning column of text */
int cursorgiven;        /* true if a cursor position was given */
                        /* it is reset at the beginning of each panel */
int adjct, adjON;       /* true if constant text is adjacent to an */
                        /* embedded field  or a type ON field is */
                        /* adjacent to constant text. */
char qsatexpr[EXPRSIZE],
    alarmexpr[EXPRSIZE],
     bellexpr[EXPRSIZE],
   cursorexpr[EXPRSIZE],
    embatexpr[EXPRSIZE],
    eraseexpr[EXPRSIZE],
     initexpr[EXPRSIZE],
     lockexpr[EXPRSIZE],
 resetmdtexpr[EXPRSIZE],
     waitexpr[EXPRSIZE],
     readexpr[EXPRSIZE],
    sleepexpr[EXPRSIZE],
    writeexpr[EXPRSIZE],
     attrexpr[EXPRSIZE],
      varexpr[EXPRSIZE],
       flexpr[EXPRSIZE];

char *cmdnames[] = {
	"alarm",
        "bell", "cursor", "erase", "init", "lock",
	"nobell", "noerase", "noinit", "nolock",
	"noread", "noresetmdt", "nowait", "nowrite",
        "read", "resetmdt", "sleep", "wait", "write", 0
};

int input(), any(), extrachars(), lookup(), maybetrue();
char *strend(), *nextsym();

main(argc, argv)
int argc;
char *argv[];
{
	int i, qoption, pid, rpid, status;
	char *suffix;
	char *newargv[22];
	int delete[22];

	if (argc > 20)
		bomb("too many args\n");
	qoption = 0;
	totalerrors = 0;
	for (i = 0; i <= argc-1; ++i)
		delete[i] = 0;
	for (i = 1; i <= argc-1; i++) {
		newargv[i] = argv[i];
		if (strcmp(argv[i], "-q") == 0)
                        qoption = 1;
		else {
                        suffix = strend(newargv[i])-2;
                        if (strcmp(suffix, ".q") == 0) {
                                strcpy(qname, newargv[i]);
                                strcpy(suffix, ".c");
                                strcpy(cname, newargv[i]);
				errors = 0;

				quickscreen();

				totalerrors += errors;
                                delete[i] = 1;
			}
		}
	}
	newargv[0] = "cc";
	newargv[argc] = "-lq";
	newargv[argc+1] = 0;

	if (totalerrors > 0)
		exit(1);
	else if (qoption)
		exit(0);
	else {
		if ((pid = fork()) < 0)
			bomb("cannot fork\n");
		if (pid == 0) {
			execv("/bin/cc", newargv);
			bomb("cannot exec /bin/cc\n");
		}
		while ((rpid = wait(&status)) != pid && rpid >= 0)
			;
		if (rpid < 0)
			bomb("no child to wait for\n");
		if (status != 0)
			exit(1);
		else {
                        for (i = 1; i <= argc-1; i++)
                                if (delete[i])
                                        unlink(newargv[i]);
                        exit(0);
		}
	}
}

bomb(args)
int *args;
{
	fprintf(stderr, "%r", &args);
	exit(1);
}

/*
 * preprocess C source code.
 * replace QUICKSCREEN constructs with C statements.
 */

quickscreen()
{
        if ((dotq = fopen(qname, "r")) == NULL) {
		++errors;
                fprintf(stderr, "%s: cannot open\n", qname);
                return;
        }
        if ((dotc = fopen(cname, "w")) == NULL) {
		++errors;
                fprintf(stderr, "%s: cannot open\n", cname);
		fclose(dotq);
                return;
        }
	qline = 0;
	cline = 0;
	*qsatexpr = 0;
	output("#include <qsdefs.h>\n");
	output("#line 1 \"%s\"\n", qname);
	while (input(0) != EOF) {
		nextsym();
		if (strcmp(symbol, "panel") == 0)
			panelproc();
		else if (strcmp(symbol, "at") == 0)
			atproc();
		else if (strcmp(symbol, "erase") == 0)
			eraseproc();
		else {
			if (escnl)
				output("#line %d \"%s\"\n", qline, qname);
			output("%s\n", line);
		}
	}
        fclose(dotq);
        fclose(dotc);
}

/*
 * read next line from the .q file into 'line'.
 * don't let \(newline) terminate the line
 * but remember if a newline was escaped.
 * expand tabs to spaces if requested. tabs are standard: 9, 17, 25, ...
 * set lp to the first non-white character in line.
 * increment qline.
 * return EOF if end of file else !EOF.
 */

input(expandtabs)
int expandtabs;
{
#       define EOFA 255
	int last;

	escnl = 0;
	lp = line-1;
	while ((*++lp = getc(dotq)) != '\n' && *lp != EOFA)
		if (*lp == '\\') {
			if ((*++lp = getc(dotq)) == '\n') {
				escnl = 1;
				lp -= 2;
				++qline;
			}
		} else if (expandtabs && *lp == '\t') {
			*lp = ' ';
			last = lp + (8 - (lp-line)%8) - 1;
			while (lp < last)
				*++lp = ' ';
		}
        if (*lp == EOFA)
		return(EOF);
	else {
                *lp = 0;
		lp = line;
		while (*lp == ' ' || *lp == '\t')
			++lp;
		++qline;
		return(!EOF);
        }
}

/*
 * print on the .c file and increment 'cline'.
 * if there are errors further output would never be used
 * so just return.
 */

output(args)
int *args;
{
	if (errors > 0)
		return;
	fprintf(dotc, "%r", &args);
	++cline;
}

/*
 * skip white space.
 * get the next symbol in 'line' and put it in 'symbol'.
 * a symbol is either a series of letters or a single non-letter.
 * later checking can either use
 * strcmp(symbol, "panel") or *symbol == '{'
 * leave 'lp' pointing one beyond the last char copied.
 * return symbol.
 */

char *
nextsym()
{
	char *sp;

	while (*lp == ' ' || *lp == '\t')
		++lp;
	sp = symbol;
	if (isalpha(*lp))
		while (isalpha(*lp))
			*sp++ = *lp++;
	else if (*lp)
		*sp++ = *lp++;
	*sp = 0;
	return(symbol);
}

/*
 * process an "at" construct
 */

atproc()
{
	trimcomment();
	getexpr(qsatexpr, ";");
	if (*lp == ';') {
		++lp;
                if (extrachars())
                        error("extra chars after at construct\n");
                else {
                        BEGIN_QS_LINES;
                        output("qsat(%s);\n", qsatexpr);
                        END_QS_LINES;
		}
	}
}

/*
 * fix up the end of 'line'.
 * remove any trailing white space.
 * if there is a comment truncate it.
 * remove any more trailing white space.
 * don't use lp for scanning.
 */

trimcomment()
{
	char *p, last;

	p = strend(line)-1;
	last = line+1;
	while (p >= line && (*p == ' ' || *p == '\t'))
		--p;
	if (p >= last && *p == '/' && *(p-1) == '*') {
		p -= 2;
		while (p >= last && !(*p == '*' && *(p-1) == '/'))
			--p;
		if (p >= last)
			p -= 2;
	}
	while (p >= line && (*p == ' ' || *p == '\t'))
		--p;
	*++p = 0;
}

/*
 * scan 'line' for an expression until you get to any character in t.
 * copy the expression into s.
 * count parentheses and stop only if they match.
 * stop on a null if you get one.
 * you cannot have a # in an expression.
 * you cannot have a , in an expression outside of parentheses.
 * leave 'lp' ON, not beyond, the char in t that stopped you.
 */

getexpr(s, t)
char *s, *t;
{
	int n, paren;

	n = 0;
	paren = 0;
	while (*lp) {
		if (any(*lp, t) && paren == 0) {
			*s = 0;
			return;
		} else if (paren == 0 && (*lp == '#' || *lp == ',')) {
			error("expression syntax\n");
			return;
		} else if (*lp == '(')
			++paren;
		else if (*lp == ')')
			--paren;

		*s++ = *lp++;

		if (++n >= EXPRSIZE) {
			error("expression longer than %d chars\n",
                              EXPRSIZE);
			return;
		}
	}
	error("expression not terminated\n");
}

/*
 * are there any more chars on the line aside from white space?
 * 'lp' is pointing where nextsym left it - one beyond the last symbol.
 */

int
extrachars()
{
	while (*lp == ' ' || *lp == '\t')
		++lp;
	return(*lp);
}

/*
 * print an error on the stderr and increment 'errors'.
 * the error includes the .q name and the line number in the .q file.
 */

error(args)
int *args;
{
	fprintf(stderr, "%s:%d: %r", qname, qline, &args);
	++errors;
}

/*
 * process an "erase" construct
 */

eraseproc()
{
	trimcomment();
	if (strcmp(nextsym(), "input") != 0)
		error("no 'input' after 'erase'\n");
	else if (strcmp(nextsym(), ";") != 0)
		error("no ';' after 'input'\n");
	else if (extrachars())
		error("extra chars after erase construct\n");
	else {
                BEGIN_QS_LINES;
                output("qsbfinit(1, 0, 0, 0, 0);\n");
		output("qswrite();\n");
		output("qsread(1);\n");
                END_QS_LINES;
	}
}

/*
 * process a "panel" construct
 */

panelproc()
{
	trimcomment();
	commands();

	BEGIN_QS_LINES;
	if (maybetrue(initexpr))
                output("qsbfinit(0, %s, %s, %s, %s);\n",
                       eraseexpr, bellexpr, lockexpr, resetmdtexpr);

	if (extrachars())
		error("extra chars on panel line\n");
	if (*symbol == '{')

		text_handler();

	else if (*symbol != ';')
		error("panel line should end with ; or {\n");

	if (*cursorexpr) {
		output("qsbfsba(%s);\n", cursorexpr);
		output("qsbfic();\n");
	}
	if (maybetrue(writeexpr))
	        output("qswrite();\n");
	if (*sleepexpr)
		output("sleep(%s);\n", sleepexpr);
	if (*alarmexpr) {
		output("alarm(%s);\n", alarmexpr);
		output("qsalarm = 0;\n");
		/*
		 * the line below is equivalent to
                 *  signal(SIGALRM, qscatchalarm);
		 * but does not need #include <signal.h>
		 */
		output("signal(14, qscatchalarm);\n");
	}
	if (maybetrue(readexpr))
	        output("qsread(%s);\n", waitexpr);
	if (*alarmexpr) {
		output("if (!qsalarm)\n\t");
		/*
		 * signal(SIGALRM, SIG_IGN);
		 */
		output("signal(14, 1);\n");
	}
	END_QS_LINES;
}

/*
 * processes panel commands.
 * 'symbol' now contains "panel".
 * Upon return, 'symbol' should contain either '{' or ';'.
 */

commands()
{
	/*
	 * mdtresetcmd rather than resetmdtcmd because of
	 * C's limit of 8 for identifiers.
	 */
	int bellcmd, erasecmd, initcmd, lockcmd,
            readcmd, mdtresetcmd, waitcmd, writecmd;

	/*
	 * first set the defaults.
	 */
              *alarmexpr  =  0;
	strcpy(bellexpr,    "0");
	      *cursorexpr =  0;
	strcpy(eraseexpr,   "1");
	strcpy(initexpr,    "1");
	strcpy(lockexpr,    "0");
	strcpy(readexpr,    "1");
	strcpy(resetmdtexpr, "1");
              *sleepexpr  =  0;
	strcpy(waitexpr,    "1");
	strcpy(writeexpr,   "1");
	bellcmd = erasecmd = initcmd = lockcmd = 0;
        readcmd = mdtresetcmd = waitcmd = writecmd = 0;
	cursorgiven = 0;

	nextsym();
	if (*symbol != '(')
		return;
	nextsym();
	while (*symbol && *symbol != ')') {
		switch (lookup()) {
		case ALARM:
			if (*alarmexpr) {
				error("alarm expression already given\n");
				skiperror();
			} else {
                                nextsym();
                                if (*symbol != '=') {
                                        error("no '=' after alarm\n");
                                        skiperror();
                                } else {
                                        getexpr(alarmexpr, ",)");
                                        *symbol = *lp++;
                                }
			}
			break;
		case BELL:
			optional(bellexpr);
			++bellcmd;
			break;
		case CURSOR:
			if (cursorgiven) {
				error("cursor position already given\n");
				skiperror();
			} else {
                                nextsym();
                                if (*symbol != '=') {
                                        error("no '=' after cursor\n");
                                        skiperror();
                                } else {
                                        getexpr(cursorexpr, ",)");
                                        *symbol = *lp++;
					cursorgiven = 1;
                                }
			}
			break;
		case ERASE:
			optional(eraseexpr);
			++erasecmd;
			break;
		case INIT:
			optional(initexpr);
			++initcmd;
			break;
		case LOCK:
			optional(lockexpr);
			++lockcmd;
			break;
                case NOBELL:
			strcpy(bellexpr, "0");
			nextsym();
			++bellcmd;
			break;
		case NOERASE:
			strcpy(eraseexpr, "0");
			nextsym();
			++erasecmd;
			break;
		case NOINIT:
			strcpy(initexpr, "0");
			nextsym();
			++initcmd;
			break;
		case NOLOCK:
			strcpy(lockexpr, "0");
			nextsym();
			++lockcmd;
			break;
		case NORESETMDT:
			strcpy(resetmdtexpr, "0");
			nextsym();
			++mdtresetcmd;
			break;
		case NOWAIT:
			strcpy(waitexpr, "0");
			nextsym();
			++waitcmd;
			break;
		case NOREAD:
			strcpy(readexpr, "0");
			nextsym();
			++readcmd;
			break;
		case NOWRITE:
			strcpy(writeexpr, "0");
			nextsym();
			++writecmd;
			break;
		case READ:
			optional(readexpr);
			++readcmd;
			break;
		case RESETMDT:
			optional(resetmdtexpr);
			++mdtresetcmd;
			break;
		case SLEEP:
			if (*sleepexpr) {
				error("sleep expression already given\n");
				skiperror();
			} else {
                                nextsym();
                                if (*symbol != '=') {
                                        error("no '=' after sleep\n");
                                        skiperror();
                                } else {
                                        getexpr(sleepexpr, ",)");
                                        *symbol = *lp++;
                                }
			}
			break;
		case WAIT:
			optional(waitexpr);
			++waitcmd;
			break;
		case WRITE:
			optional(writeexpr);
			++writecmd;
			break;
                default:
                        error("unknown command: %s\n", symbol);
                        skiperror();
                        break;
                }
		if (*symbol == ')')
			;
		else {
			if (*symbol != ',') {
                                error("no , or ) after command\n");
                                skiperror();
			}
			if (*symbol == ',')
                                nextsym();
		}
        }
        if (bellcmd >= 2 || erasecmd >= 2   || initcmd >= 2 ||
            lockcmd >= 2 || mdtresetcmd >=2 || readcmd >= 2 ||
            waitcmd >= 2 || writecmd >= 2                     )
                error("duplicate command\n");
	if (*symbol == ')')
		nextsym();
	else
		error("missing ) after commands\n");
}

/*
 * find 'symbol' in the list 'cmdnames'.
 * if you found it return where otherwise -1.
 */

int
lookup()
{
	int r;
	char **p;

	p = cmdnames;
	while (*p)
		if ((r = strcmp(symbol, *p)) == 0)
			return(p-cmdnames);
		else if (r > 0)
			++p;
		else
			return(-1);
	return(-1);
}

/*
 * if the next symbol is an '=', get an expression into s
 * otherwise copy "1" there.
 */

optional(s)
char *s;
{
	nextsym();
	if (*symbol == '=') {
		getexpr(s, ",)");
	        *symbol = *lp++;
	} else
	        strcpy(s, "1");
}

/*
 * this is used to skip past a syntax error.
 * advance 'lp' until you get a ',' or a ')'.
 * count parentheses and stop only if they match.
 * put the char that you stopped on into 'symbol'.
 * leave 'lp' one past that char.
 */

skiperror()
{
	char c;
	int paren;

	if (*symbol == ',' || *symbol == ')')
		return;
	paren = 0;
	while (c = *lp++)
		if ((c == ',' || c == ')')  && paren == 0) {
			*symbol = c;
			return;
		} else if (c == '(')
			++paren;
		else if (c == ')')
			--paren;
	*symbol = 0;
}

/*
 * if the char array expr is "0" or "1" return 0 or 1.
 * otherwise output an if statement that evaluates the expression
 * and return 1 (maybe).
 * this function is used to make the init, write, and wait conditional.
 * a tab is put out at the last so that the indentation looks good.
 */

int
maybetrue(expr)
char *expr;
{
	if (strcmp(expr, "0") == 0)
		return(0);
	else if (strcmp(expr, "1") == 0)
		return(1);
	else {
	        output("if (%s)\n\t", expr);
	        return(1);
	}
}

/*
 * processes panel text.
 * quits when it finds the closing brace after the text.
 */

text_handler()
{
	if (input(1) == EOF) {
		error("panel text terminated prematurely\n");
		return;
	}
	trimcomment();
	fixembeds();
	/*
	 * if there is text and
         *    it does not begin with an embedded at then
	 *       initialize the "paperlmrgn" runtime variable.
	 */
	if (*lp != '}' &&
	    !(*lp == '#' && *(lp+1) == '@'))
		output("qstxtinit(%d);\n", lp-line+1);
	/*
	 * process lines of text
	 */
	skips = 0;
	while (*lp != '}') {
		if (line[0] == 'C')      /* C statement within panel */
			output("%s\n", line+1);
		else if (*lp)
			nonblank_line();
		else if (strcmp(eraseexpr, "1") == 0)
			++skips;
		else
			output("qsbfra();\n");
                if (input(1) == EOF) {
                        error("panel text terminated prematurely\n");
                        return;
                }
	        trimcomment();
	}
	/*
	 * any extra chars aside from an optional ';'?
	 */
	++lp;
	while (*lp == ' ' || *lp == '\t')
		++lp;
	if (*lp == ';')
		++lp;
	if (extrachars())
		error("extra chars after }\n");
}

/*
 * process a non-blank line of text in a panel construct
 */

nonblank_line()
{
	fixembeds();
	adjct = adjON = 0;
	dosba();
	while (*lp)
		if (*lp == '#')
			if (*(lp+1) == '@')
				dosba();
			else {
				embeddedfield();
				afterfield();
			}
		else
			constanttext();
}

/*
 * fix up embedded items.
 * if the first char after the # is blank SHIFT the chars inside
 * the item so that the first char is non-blank.
 * #   hello   there # becomes #hello   there    #
 * do not use 'lp'.
 * it now points to the first non-white char in 'line'.
 */

fixembeds()
{
        char *p, *q;

        p = lp;
        while (*p)
                if (*p != '#')
                        ++p;
                else {
			++p;
			q = p;
			while (*p == ' ')
				++p;
			while (*p && *p != '#')
				*q++ = *p++;
			while (q < p)
				*q++ = ' ';
			if (*p == '#')
				++p;
		}
}

/*
 * output a call for an SBA for the beginning of the line
 * or for an embedded at.
 * 'lp' now points to first non-white char in 'line'.
 */

dosba()
{
	if (*lp == '#' && *(lp+1) == '@') {  /* embedded at */
		emb_at_handler();
		skips = 0;
	} else
		begincol = lp-line+1;
	--begincol;                           /* for the attribute char */
        output("qsbflinit(%d, %d);\n", skips, begincol);
        skips = 0;
}

/*
 * process one or more(?) contiguous embedded at's.
 * output calls to update paperlmrgn, row, and screenlmrgn
 * runtime variables.
 * set begincol to the column of the last INITIAL #.
 * leave 'lp' one char past the last FINAL #.
 */

emb_at_handler()
{
        while (*lp == '#' && *(lp+1) == '@') {
                begincol = lp-line+1;
                lp += 2;
		getexpr(embatexpr, "#");
                if (*lp == '#')
                        ++lp;
        }
        output("qstxtinit(%d);\n", begincol);
        output("qsat(%s);\n", embatexpr);
}

/*
 * 'lp' is now pointing at the initial # of an embedded field.
 * get the attribute, variable, and fieldlength into expressions.
 * output a qsbuffld statement.
 *
 * an embedded field can have several forms:
 * #hello#   #1hello#    #1hello, 5#    #1, hello, 5#
 * #OH, "high intensity"#   #IN, hello#
 * #i+4, (j == 2)? hello: goodbye, k+(l/4)#
 */

embeddedfield()
{
        ++lp;
        if ('1' <= *lp && *lp <= '9') {
                attrexpr[0] = *lp;
		if ('1' <= *lp && *lp <= '3' &&
                    '1' <= *(lp+1) && *(lp+1) <= '9') {
			++lp;
			attrexpr[1] = *lp;
			attrexpr[2] = 0;
		} else
	                attrexpr[1] = 0;
                if (*lp >= '7')
                        if (cursorgiven)
                                error("cursor position already given\n");
                        else
                                cursorgiven = 1;
                /*
                 * optional comma?
                 */
                if (*(lp+1) == ',')
                        ++lp;
        } else {
                getexpr(attrexpr, ",#");
                if (*(strend(attrexpr)-1) == 'C')
                        if (cursorgiven)
                                error("cursor position already given\n");
                        else
                                cursorgiven = 1;
        }
	if (*lp == '#') {
		strcpy(varexpr, attrexpr);
		strcpy(attrexpr, "1");
		strcpy(flexpr, "0");
		++lp;
	} else {
                ++lp;
                getexpr(varexpr, ",#");
                if (*lp == ',') {
                        ++lp;
                        getexpr(flexpr, "#");
                } else
                        strcpy(flexpr, "0");
                ++lp;
	}
	if (adjct) {
                if (strcmp(attrexpr, "ON") == 0 ||
		    strcmp(attrexpr, "1") == 0     )
			strcpy(attrexpr, "0");
		else
			error("constant text adjacent to non-ON field\n");
		adjct = 0;
	}
        output("qsbuffld(%s, %s, %s);\n",
                attrexpr, varexpr, flexpr);
}

/*
 * if the last field might have been an input field
 * and it is NOT followed by an embedded field or constant text,
 * delimit it with a null output field.
 * 'lp' now points one char after the final # of an embedded item.
 * adjust 'lp' and check for spaces between two embedded fields and
 * between embedded fields and constant text.
 */

afterfield()
{
	char a, b, c;
	int lastwasinput, followed;

	a = *attrexpr;
	b = *(strend(attrexpr)-1);
	lastwasinput = !((isdigit(a) && '1' <= b && b <= '3') ||
                         a == 'O'
                        );
	a = lp[0];
	b = lp[1];
	c = lp[2];
	followed = 1;
	if (!a)                                         /* #$           */
		followed = 0;
	else if (a == '#' && b == '@')                  /* ##@210#      */
		followed = 0;
	else if (a == '#' && b != '@')                  /* ##1age, 5#    */
		error("adjacent embedded fields\n");
	else if (a != ' ') {                            /* #hello       */
		if (strcmp(attrexpr, "ON") == 0 ||
		    strcmp(attrexpr, "ONN") == 0 ||
		    strcmp(attrexpr, "1")  == 0 ||
		    strcmp(attrexpr, "0")  == 0   )
			adjON = 1;
		else
			error("non ON-field adjacent to constant text\n");
	} else {
                if (b != '#')                           /* # hello      */
                        ++lp;
                else if (c != '@')                      /* # #1age, 5#   */
                        ++lp;
                else {                                  /* # #@210#     */
                        ++lp;
                        followed = 0;
                }
	}
	if (lastwasinput && !followed)
		output("qsbuffld(1, \"\", 0);\n");
}

/*
 * 'lp' is now pointing to the first char of a constant text field.
 * copy the text into ctbuf. the end of text is indicated either by
 * the end of the line, an embedded at, or an embedded field.
 * blanks at the end of constant text and text adjacent to an
 * embedded field need special handling.
 * output a call to qsbuffld() to display the constant text.
 *
 * leave 'lp' pointing after the last constant text character.
 */

constanttext()
{
	int moretext;
	char *ctp;
	static char ctbuf[LINESIZE];

	ctp = ctbuf;
	moretext = 1;
	while (moretext) {
                /*
                 * a " must be escaped in ctbuf because constant
                 * text will appear between "s in the .c file.
                 */
                if (*lp == '"')
                        *ctp++ = '\\';
                *ctp++ = *lp++;
                if (!*lp)
                        moretext = 0;
                else if (*lp == '#') {
                        moretext = 0;
			if (*(ctp-1) == ' ')
				--ctp;
                        else if (*(lp+1) != '@')
                                adjct = 1;
		}
	}
        *ctp = 0;
        output("qsbuffld(%d, \"%s\", 0);\n", (adjON)? 0: 1, ctbuf);
	adjON = 0;
}
