#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/dir.h>
#include <sys/stat.h>

#define T3270           1
#define TASCII          2

#define E1PR            'e'             /* format to print on E1 printer */
#define PRINT           'p'             /* format to print on TN printer */
#define QUME            'q'             /* format to print on Qume */
#define QUME3270        'Q'             /* Qume format to display on 3270 */
#define CAT             'c'             /* cat the output; don't ned it */
#define FILENAME        'f'             /* direct filename, no searching */
#define MERGE           'M'             /* flag from the mman command */

#define HYPHEN          '-'
#define COMMA           ','
#define COLON           ':'
#define PERIOD          '.'
#define SLASH           '/'
#define EOS             '\0'

#define DOTi            ".i"
#define TMP             "/tmp"
#define NULLSTR         ""
#define DOT             "."
#define DOTDOT          ".."
#define MANPF           "/.../man"

#define FIRSTVOL        1
#define LASTVOL         3

char    man1[] = "/usr/man/man1";
char    man2[] = "/usr/man/man2";
char    man3[] = "/usr/man/man3";

char    vol1[] = "/usr/man/vol1";
char    vol2[] = "/usr/man/vol2";
char    vol3[] = "/usr/man/vol3";

/*
*  Structure defining the directories to search
*/
struct dirs {
	char    *d_src;                 /* source directory name */
	char    *d_out;                 /* output directory name */
	struct  dirs    *d_next;        /* next dirs structure */
};

struct dirs sysdirs[LASTVOL + 1] = {
	NULL, NULL, NULL,               /* header node */
	man1, vol1, NULL,               /* dirs for volume 1 */
	man2, vol2, NULL,               /* dirs for volume 2 */
	man3, vol3, NULL,               /* dirs for volume 3 */
};
struct  dirs    *hdirs = &sysdirs[0];

char    *format = "-rr0";
char    *devtbl = "-Tq3270";
char    *filter = "tprf -l22";
char    *mman = "";
char    *usage = "usage: %s [options] title ...\n";


char    msg[256] = "";
char    *longdash = "--------------------------------"; /* 32 dashes */
char    pflg = 0;               /* TN printer */
char    qflg = 0;               /* qume */
char    cflg = 0;               /* cat on standard output */

FILE    *pfd = NULL;

main(argc, argv)
int     argc;
char    **argv;
{
	char    *cmdname;
	char    **arg1 = &argv[1];
	int     term;

	term = termtype();
	if (term & TASCII)
		cflg++;
	dolman();
	cmdname = *argv++;
	argc--;
	while (argc--) {
		if (**argv == HYPHEN) {
			switch (*++(*argv)) {
			case E1PR:
				format = "-rr4";
				devtbl = "-Te1pr";
				filter = "oprf | e1opr";
				pflg++;
				break;
			case PRINT:
				format = "-rr4";
				devtbl = "-Ttn";
				filter = "oprf | opr -c U";
				pflg++;
				break;
			case QUME:
				format = "-rr4";
				devtbl = "-Tq12";
				filter = "qprf";
				qflg++;
				break;
			case QUME3270:
				format = "-rr4";
				filter = "tprf -w102 -o-9";
				qflg++;
				break;
			case CAT:
				cflg++;
				break;
			case FILENAME:
				argv++;
				if (argc--)
					nosearch(*argv);
				break;
			default:
				if (isdigit(**argv)) {
					addsysdir(0);
					while (isdigit(**argv))
						addsysdir(*(*argv)++ - '0');
				}
				else {
					fprintf(stderr, usage, cmdname);
					exit(1);
				}
				break;
			}
		}
		else
			loctitles(arg1, *argv);
		argv++;
	}
	if (pfd != stdout && pfd != NULL) {
		fprintf(pfd, "%s The  End %s\n", longdash, longdash);
		pclose(pfd);
	}
	fprintf(stderr, "%s", msg);
}
/*
 * addsysdir - add a system directory to the list of those to be searched
 */
addsysdir(vol)
int     vol;
{
	struct  dirs   *newdir;
	struct  dirs   *d;
	int     i;

	if (vol == 0) {
		for (d = hdirs; d->d_next; d = d->d_next)
			for (i = FIRSTVOL; i <= LASTVOL; i++)
				if (d->d_next == &sysdirs[i]) {
					d->d_next = NULL;
					return;
				}
		return;
	}

	if (vol >= FIRSTVOL && vol <= LASTVOL) {
		newdir = &sysdirs[vol];
		for (d = hdirs; d->d_next; d = d->d_next)
			if (d->d_next == newdir)
				return;
		d->d_next = newdir;
		newdir->d_next = NULL;
	}
}
/*
 * dolman - process the $MAN shell variable
 * MAN=[n][:srcdir[,outdir]...]
 */
dolman()
{
	struct  dirs    *newdir;
	struct  dirs    *d;
	char    *s, *saves, *b;
	char    *getenv(), *malloc();

	if ((s = getenv("MAN")) == NULL) {
		addsysdir(1);
		return;
	}
	if (!*s || *s == COLON)
		addsysdir(1);           /* default volume */
	else
		while (*s && *s != COLON)
			addsysdir(*s++ - '0');
	/*
	*  Get the source and output directories
	*/
	while (*s++) {
		newdir = (struct dirs *) malloc(sizeof(*newdir));
		saves = s;
		while (*s && *s != COMMA && *s != COLON)
			*s++;
		b = newdir->d_src = malloc((unsigned) (s - saves + 1));
		strncpy(b, saves, s - saves);
		b[s - saves + 1] = EOS;
		if (*s == COMMA) {
			saves = ++s;
			while (*s && *s != COLON)
				s++;
			b = newdir->d_out = malloc((unsigned) (s - saves + 1));
			strncpy(b, saves, s - saves);
			b[s - saves + 1] = EOS;
		}
		else
			newdir->d_out = TMP;

		for (d = hdirs; d->d_next; d = d->d_next)
			if (strcmp(newdir->d_src, d->d_src) == 0)
				continue;
		newdir->d_next = hdirs->d_next;
		hdirs->d_next = newdir;
	}
}
/*
 * loctitles - locate the titles
 * Check for duplicate titles, and then search each directory
 */
loctitles(arg1, name)
char    **arg1;
char    *name;
{
	struct  dirs    *d;
	int     n = 0;
	FILE    *fd,    *openf();

	while (*arg1 != name) {
		if (strcmp(*arg1, name) == 0)
			return;                 /* duplicate title */
		arg1++;
	}

	for (d = hdirs->d_next; d; d = d->d_next) {
		if ((fd = fopen(d->d_src, "r")) == NULL)
			continue;
		n += searchdir(fd, name, d);
		fclose(fd);
	}

	if (n == 0)
		sprintf(msg, "%sCannot find title %s\n", msg, name);
}
/*
 * searchdir - search a directory for a title
 * returns:  the number of matches found
 */
searchdir(fd, name, cdirs)
FILE    *fd;
char    *name;
struct  dirs    *cdirs;
{
	struct  direct  dir;
	char    filename[64];
	int     len;
	int     n = 0;
	FILE    *ifd;

	while (fread((char *)&dir, sizeof(dir), 1, fd) == 1) {
		if (dir.d_ino == 0 ||
		    strcmp(dir.d_name, DOT) == 0 ||
		    strcmp(dir.d_name, DOTDOT) == 0)
			continue;

		len = strlen(name);
		if (strncmp(name, dir.d_name, len) == 0) {
			if (strcmp(&dir.d_name[len], DOTi) == 0) {
				n++;
				sprintf(filename, "%s/%s", cdirs->d_src,
					dir.d_name);
				ifd = openf(filename, "r");
				while (fscanf(ifd, "%s", filename) != EOF)
					nroff(filename, cdirs, name);
				fclose(ifd);
			}
			else if (dir.d_name[len] == PERIOD || dir.d_name[len] == EOS) {
				nroff(dir.d_name, cdirs, name);
				n++;
			}
		}
	}
	return(n);
}
/*
 * Check the flags and dates of the source and output files and
 * nroff the source if necessary.
 */
nroff(fname, cdirs, argname)
char    *fname;
struct  dirs    *cdirs;
char    *argname;
{
	struct  stat    srcstat, outstat, sbuf;
	static  char    first = 1;
	char    srcfn[100], outfn[100], cmd[200], umanpf[100];
	char    *manpf = "/usr/lib/man.browse";
	char    *home, *getenv();
	FILE    *popen();
	extern int errno;

	if (strcmp(cdirs->d_src, "") != 0)
		sprintf(srcfn, "%s/%.14s", cdirs->d_src, fname);
	else
		strcpy(srcfn, fname);
	if (strcmp(cdirs->d_out, TMP) != 0)
		sprintf(outfn, "%s/%.14s", cdirs->d_out, fname);
	else
		sprintf(outfn, "%s/man.%d", cdirs->d_out, getpid());

	if (pflg)
		sprintf(cmd, "/usr/lib/ss/fonts %s | tbl -TX | nroff -man %s %s | %s -n %s.man",
			srcfn, format, devtbl, filter, argname);
	else if (qflg)
		sprintf(cmd, "/usr/lib/ss/fonts %s | tbl | nroff -man -e %s %s | %s",
			srcfn, format, devtbl, filter);
	else {
		if (stat(srcfn, &srcstat) == -1) {
			fprintf(stderr, "%s: unable to open\n", srcfn);
			return;
		}

		if (stat(outfn, &outstat) == -1 || srcstat.st_mtime > outstat.st_mtime) {
			if (access(outfn, 2) != -1 || access(cdirs->d_out, 2)!= -1) {
				sprintf(cmd, "/usr/lib/ss/fonts %s | tbl | nroff -man -e %s %s %s | %s > %s",
					srcfn, format, devtbl, mman, filter, outfn);
				system(cmd);
			}
		}

		if (!cflg) {
			if (first) {
				if ((home = getenv("HOME")) != NULL) {
					strcpy(umanpf, home);
					strcat(umanpf, MANPF);
					if (stat(umanpf, &sbuf) != -1)
						manpf = umanpf;
				}
				sprintf(cmd, "ned - -r -F %s", manpf);
				pfd = popen(cmd, "w");
			}
		} else
			pfd = stdout;

		cat(outfn, pfd);
		first = 0;
		if (strncmp(outfn, TMP, sizeof(TMP)-1) == 0)
			unlink(outfn);
		return;
	}
	system(cmd);
}
/*
 * cat - cat a file to the standard output
 */
cat(filename, ofd)
char    *filename;
FILE    *ofd;
{
	int     c;
	FILE    *ifd;
	FILE    *openf();

	ifd = openf(filename, "r");
	while ((c = getc(ifd)) != EOF)
		putc(c, ofd);
	fclose(ifd);
}
/*
 * openf - fopen with error checking
 */
FILE    *
openf(fn, mode)
char    *fn;
char    *mode;
{
	FILE    *fd;

	if ((fd = fopen(fn, mode)) == NULL) {
		fprintf(stderr, "%s: unable to open\n", fn);
		exit(1);
	}
	return(fd);
}
/*
 * nosearch - an argument with the -f option to explicitly man
 * a file.  No searching of system or user directories is done.
 */
nosearch(name)
char    *name;
{
	struct  dirs    *newdir;
	struct  stat    statbuf;
	char    *malloc();

	if (stat(name, &statbuf) == -1)
		sprintf(msg, "%sCannot find title %s\n", msg, name);
	else {
		newdir = (struct dirs *) malloc(sizeof(struct dirs));
		newdir->d_src = (*name == SLASH)? NULLSTR: DOT;
		newdir->d_out = TMP;
		newdir->d_next = NULL;
		nroff(name, newdir, name);
	}
}
/*
 * termtype - does the best it can to determine the type of terminal
 */
termtype()
{
	char    *tty;
	char    *ttyname();
	struct  stat    statb;

	tty = ttyname(2);
	/*
	 *  /dev/ttyxxx -> /dev/tubxxx
	 *  0123456789
	 */
	tty[5] = 't';
	tty[6] = 'u';
	tty[7] = 'b';
	if (stat(tty, &statb) == -1)
		return(TASCII);
	return(T3270);
}
