/*
 * main, print routines for debugger
 */

#include "dcon.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <setjmp.h>
#include <stdioerr.h>

jmp_buf env;

FILE *namelist;
FILE *out = stdout;
FILE *in = stdin;
FILE *infile[NUMINFIL] = { stdin };
int inindex = 0;
FILE *filptr;
char *objname = "core";
char *nmname = "a.out";
int prtmode 'x';
int prtlen BPW;
int savplen 0;
int instlen 0;
int shared 0;
int pmap 0;
int nomap 0;
int xlate 0;
int fildes;		/* core file's fd */
int dot 0;
int dotdot 0;
int textsize 0;
int offset 0;
int fend 0;
int stacksize 0;
int dim = -1;
int errflg = 0;
int argsflag = 0;
int decl = 0;
int inpattern = 0;
int quitread = 0;
char pattern[MAXPATTERN];
SYM *hashp[SYMSIZ];
SYM *valp[SYMSIZ];

static SYM *curfunc = NULL;

/*
 * main routine - doesn't do much, just figures out
 *	what kind of file is being examined, reads the name list
 *	and then calls yyparse().
 * the magic flags that are undocumented are helpful for
 * debugging the debugger
 *
 *	-m	means display the result of the map function so you
 *		can see what location is actually being examined
 *	-r	forces the file to be read in raw mode (no mapping)
 *	-d	sets the Yacc debugging variable yydebug so that all sorts
 *		of garbage pours out at you while parsing the input
 */
main(argc, argv)
int argc;
char **argv;{

	extern int yydebug;
	int s, catchsig(), qcmp();
	struct stat sbuf;

	setjmp(env);
	onerror(EBADUSE, EIGNORE);
	for (s = 1; s < 4; s++)
		signal(s, catchsig);
	setbuf(stdout, NULL);
	while (*argv[1] == '-'){
		switch(*(argv[1]+1)){
			case 'm':
				pmap++;
				break;
			case 'r':
				nomap++;
				break;
			case 'd':
				yydebug = -1;
				break;
			default:
				error(1, "Unknown option: %s\n", argv[1]);
			}
		++argv;
		--argc;
		}
	switch(argc){
		case 3:
			objname = argv[1];
			nmname = argv[2];
			break;
		case 2:
			nmname = objname = argv[1];
			break;
		default:
			error(1, "Usage:  dcon file [namelist]");
		}
	if ((fildes = open(objname, 2)) == -1){
		if ((fildes = open(objname, 0)) == -1){
			error(1, "Can't read %s", objname);
			}
		}
	if (stat(objname, &sbuf) == -1){
		error(1, "Can't determine type of %s", objname);
		}
	if (seek(fildes, 0, 2) == -1) fend = INFINITY;
	else fend = tell(fildes);
	if (!nomap && (s = (sbuf.st_mode&S_IFMT)) != S_IFCHR && s != S_IFBLK){
		if ((s = peek(0)) == A_MAGIC1 || s == A_MAGIC2) /* normal or shr */
			offset = AOUTSTART;
		else {
			offset = CORESTART;
			stacksize = readcore(fildes);
			if (stacksize < 0 || stacksize > fend) {
				offset = 0;
				}
			}
		}
	if (pmap){
		if (offset == CORESTART) printf("core dump ");
		else if (offset == AOUTSTART) printf("object file ");
		else printf("raw file ");
		printf("(stacksize = %d, fend = %d)\n", stacksize, fend);
		}
	if ((namelist = fopen(nmname, "r")) == NULL){
		error(0, "No namelist");
		}
	else {
		readnm();
		qsort(valp, nsym, BPW, qcmp);
		fclose(namelist);
		}

	setjmp(env);
	yyparse();
	putchar('\n');
	}

/*
 * readnm - read in the name list from the a.out file
 *	skip over header, sum up text, data, and relocation size to find
 *	start of symbol table
 *
 *	the textsize has to be saved
 */

readnm(){

	register int i, skip, symsize;
	register char *p;
	int c, t, s, v;
	int lflg, fflg;
	char symname[NAMESIZE];
	struct symtab *symp;

	if ((i = getw(namelist)) == A_MAGIC2) shared++;
	else if (i != A_MAGIC1){
		error(0, "Bad namelist format");
		return;
		}
	skip = textsize = getw(namelist);
	skip += getw(namelist);
	getw(namelist);
	symsize = getw(namelist);
	getw(namelist);
	skip += getw(namelist);
	fseek(namelist, 8*BPW+skip, 0);
	for (; symsize > 0; symsize -= (NAMESIZE+2*BPW)) {
		for (p = symname; p < symname+NAMESIZE; p++)
			*p = getc(namelist);
		c = 0;
		s = 1;
		t = getw(namelist);
		v = getw(namelist);
		if (t != LOCAL && symname[0] != '$') {
			if (symname[0]&0200) {
				symname[0] &= ~0200;
				if (!lflg && t != N_TEXT) c = MOS;
				else if (t == N_TEXT) lflg = 1;
				else c = class(v);
				}
			symp = insert(symname);
			symp->sclass = c;
			symp->type = t;
			symp->size = s;
			symp->val = v;
			if (c) symp->func = curfunc;
			else symp->func = NULL;
			if (t == N_TEXT && lflg == 1) curfunc = symp;
			}
		}
	}

/*
 * figure out whether a local symbol is a register or
 * on the stack
 * I haven't figured out any way to distinguish
 * a second argument (offset 4) from a register variable
 * kept in register 4.  Currently, it defaults to the argument.
 */

class(v)
int v;{

	switch(v) {
		case 5:
		case 6:
		case 7:
			return(REGVAR);
		case 4:
			return(AUTO);
		default:
			return(AUTO);
		}
	}

/*
 * setpr - set the print length and print mode
 *	the argument is a word with the first half being the print length
 *	and the second the print mode.  If either are zero, then they are
 *	not changed.
 */

setpr(v)
int v;{

	register int x, y;
	struct twoshorts { short shortx, shorty; };

	x = v.shortx;
	y = v.shorty;
	xlate = 0;
	switch(y) {
		case 0:
			break;
		case 'd':
		case 'o':
		case 'x':
			prtlen = BPW;
			prtmode = y;
			break;
		case 'c':
			prtlen = 1;
		default:
			prtmode = y;
			break;
		}
	switch(x) {
		case 'b':
			prtlen = 1;
			break;
		case 'h':
			prtlen = BPHW;
			break;
		case 'w':
			prtlen = BPW;
			break;
		case 'l':
			prtlen = BPL;
			break;
		case 'a':
			prtlen = 0;
			break;
		}
	}

/*
 * prtval - print the value at the address according to the
 *	current print length and print mode
 *
 *	adflag when set specifies that the address should be printed
 *	out as well as the data
 *	also 'i' or 'ix' modes force the address to be printed
 */

prtval(addr, adflag)
register int addr;
int adflag;{

	long x;

	if (prtlen == 0 && !PRINST) {
		print(dot = addr);
		putc('\n', out);
		if (savplen) {
			prtlen = savplen;
			savplen = 0;
			}
		return;
		}
	errflg = 0;
	x.high = peek(map(addr));
	if (prtlen > BPW)
		x.low = peek(map(addr+BPW));
	else x.low = 0;
	if (errflg) return;
	if (adflag || pmap || PRINST) fprintf(out, "%6x ", addr);
	if (pmap) fprintf(out, "(%6x) ", map(addr));
	switch(prtmode) {
		case 'i':
		case 'j':
			dotdot = addr;
			prinst(addr, (prtmode == 'j'));
			break;
		default:
			dot = addr;
			print(x);
			break;
		}
	putc('\n', out);
	}

/*
 * print out a value according to the current print length and mode
 */

print(value)
long value;{

	register int i, ivalue;
	register char *p, *q;

	if (prtlen == 0) ivalue = value >> bPW;
	else ivalue = value >> bPB*(BPL-prtlen);
	switch(prtmode) {
	/*
	 * undocumented feature:
	 *	"/ec" means print out the character interpreting it as EBCDIC
	 *	"/ewc" for a word's worth
	 *	xlate flags this
	 */
		case 'c':
			p = &value;
			for (i = 0; i < prtlen; i++){
				if (xlate) *p = etatab[*p];
				fprintf(out, "%c", *p++);
				}
			break;
		case 's':
			for (;;) {
				p = &value;
				q = p+BPW;
				while (p < q && *p != '\0')
					fprintf(out, "%c", *p++);
				if (p < q) break;
				value.high = peek(map(dot += BPW));
				}
			break;
		case 'd':
		case 'o':
		case 'x':
			if (prtlen <= BPW) {
				p = "%d";
				p[1] = prtmode;
				fprintf(out, p, ivalue);
				}
			else {
				p = "%ld";
				p[2] = prtmode;
				fprintf(out, p, value);
				}
			break;
		case 'v':
			if ((i = vlook(ivalue, 0)) == -1)
				i = 0;
			if (i < nsym-1 && valp[i+1]->name[0] != '#') {
				if (valp[i+1]->val-ivalue < ivalue-valp[i]->val)
					i++;
				}
			ivalue -= valp[i]->val;
			fprintf(out, "%s", valp[i]->name);
			if (ivalue > 0) fprintf(out, "+%d", ivalue);
			else if (ivalue < 0) fprintf(out, "-%d", -ivalue);
			break;
		case 'i':
		case 'j':
			error(0, "print mode does not match request");
			break;
		default:
			error(0, "'%c' unknown print mode", prtmode);
		}
	}

printregs(){

	register int r;
	register PSTATE *p = &state;

	if (PRINST) {
		error(0, "'i' or 'ix' print mode makes no sense for ;r");
		return;
		}
	for (r = 0; r < 16; r++) {
		fprintf(out, "r%-2d = ", r);
		print(p->ps_gpr[r]);
		putc('\n', out);
		}
	}

printlist(x, y)
register int x, y;{

	register int inst;

	inst = PRINST ? 1 : 0;
	while (x < y) {
		prtval(x, 1);
		x += inst ? instlen : prtlen;
		}
	prtval(x, 1);
	if (inst) dotdot = x;
	else dot = x;
	}



/*
 *  push fp onto inputfile stack and modify where input comes from
 */

pushin(fp)
FILE *fp;
{
	if (++inindex == NUMINFIL)
		error(0, "infile stack overflow");
	else {
		infile[inindex] = fp;
		in = fp;
		}
}



/*
 *  pop file ptr. from inputfile stack and modify where input comes from
 */

popin()
{
	if (inindex-- > 0) {
		infile[inindex+1] = '\0';
		in = infile[inindex];
		return(1);                      /* any nonzero */
		}
	return(0);
}



/*
 * all-purpose error routine
 * basically we just do a fprintf to stderr
 * errflg is used by prtval to determine if
 *	peek got the value to be printed without any errors
 * a non-zero level indicates the error is fatal, program aborts
 */

error(level, arglist)
int level;{

	register int c;

	errflg++;
	fprintf(stderr, "%r", &arglist);
	putc('\n', stderr);
	if (level) exit(level);
	}

/*
 * here's where the caught signals (interrupt, quit, & hangup) go
 * we just print out the signal and start over reading commands
 * with setexit/reset.
 *
 * when tracing, we have to make the child ignore the signal
 * by starting it with ptrace, and then clear ps_sig.
 */

catchsig(signo)
int signo;{

	int status;

	signal(signo, catchsig);
	printf("%s\n\n", signame[signo]);
	if (trace) {
		ptrace(pid, 1, &state);
		while (wait(&status) != pid);
		ptrace(pid, 0, &state);
		state.ps_sig = 0;
		}
	longjmp(env,0);
	}
