/*
* ac - command usage accounting
*
* Synopsis: ac [-abcelntxmw] [-v n] [-d fromdate todate] [-r n]
*
* Options:
*   -c   Besides total user, system, and elapsed time for each command
*        print percentage of TOTAL time for each category for each command.
*   -e   Evaluate each command by giving the average cpu and io per
*        command, and give the elapsed/cpu and io/cpu ratios.  This option
*        overrides the -clt options.
*   -l   Separate system and user cpu time; normally they are combined.
*   -t   For each command report ratio of elapsed time to the sum of the
*        user and system cpu times.
*
*   -a   Place all command names containing unprintable characters
*        and those used only once under the name "***other".
*   -v   The next argument is a number n.  Type the name of each command
*        used n times or fewer.  Await a reply from typewriter; if it
*        begins with "y" add the command to the category "***other".
*
*   -b   Sort output by sum of user and system cpu time divided by number
*        of calls.  Default sort is by sum of user and system cpu times.
*   -n   Sort by the number of calls.
*   -x   Reverse the order of sort.
*
*   -m   Provide information for most recent month (1-31st).
*   -w   Provide information for most recent week (sat-fri).
*   -d   Specifies a certain "from" date and an optional "to" date.
*        If the "to" date is omitted, only the information for the
*        "from" date will be printed.
*   -r   Must be followed by an integer number.  The accounting
*        information will be printed for the most recent "n" days.
*/

#include <stdio.h>
#include <time.h>
#include "acct.h"
#include <sys/acctfile.h>

#define  DAYSECS  (24*60*60)
#define  BYTEPWD  4

int  aflg, bflg, cflg, eflg, lflg, nflg, rflg, tflg, vflg, xflg;
int  from, to;
int  fyr,  fday, tyr,  tday;
int  tcount, trealt, tcput, tsyst, tio;
float  zero, sixty, athousand;

char usage[]  = "usage: ac [-abcelntxmw] [-v n] [-d from to] [-r n]";
char dayfile[30];
char sfrom[10], sto[10];

struct aday aday[MAXUSERS];

struct cmds cmds[MAXCMDS];
struct cmds tots[MAXCMDS];

int index[MAXCMDS], numb, bad;


main(argc,argv) char **argv;
{
	while(--argc > 0 && *argv[1]=='-') {
		argv++;
		while(*++*argv) switch(**argv) {
		case 'a':
			aflg++;
			continue;
		case 'b':
			bflg++;
			continue;
		case 'c':
			cflg++;
			continue;
		case 'e':
			eflg++;
			continue;
		case 'l':
			lflg++;
			continue;
		case 'n':
			nflg++;
			continue;
		case 't':
			tflg++;
			continue;
		case 'x':
			xflg++;
			continue;
		case 'v':
			vflg = atoi(*++argv);
			--argc;
			goto nextarg;
		case 'm':
			lastmon();
			continue;
		case 'w':
			lastweek();
			continue;
		case 'd':
			from= any(*++argv, '/')?julian(*argv):atoi(*argv);
			--argc;
			if (*argv[1]=='-')
				goto nextarg;
			to  = any(*++argv, '/')?julian(*argv):atoi(*argv);
			--argc;
			goto nextarg;
		case 'r':
			rflg = atoi(*++argv);
			--argc;
			goto nextarg;
		default:
			printf("%s\n", usage);
			exit(1);
		}
nextarg:;
	}

	if (argc > 0) {
		printf("%s\n", usage);
		exit(1);
	}

	if (rflg || (!from && !to)) maker(rflg? rflg:7);
	maked(from, to);

	zero      = 0;
	sixty     = 60;
	athousand = 1000;

	getdata();
	if (aflg) strip(1, 0);
	if (vflg) strip(vflg, 1);
	sort();
	totals();
	output();
	/*
	 * Tell user if we skipped a whole bunch of records
	 */
	if (bad > 50)
		printf("\n%d bad command records skipped.\n", bad);
}


/*
* lastmon: fills the from-to date values with dates ranging from the
*          first of last month until the end of last month.
*/
lastmon()
{
	int t, mon;
	struct tm *tm;
	struct tm *localtime();

	t = time();
	tm = localtime(t);

	mon = tm->tm_mon;
	while (mon == tm->tm_mon) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	to = tm->tm_year*1000 + tm->tm_yday+1;

	t -= DAYSECS*25;
	while (tm->tm_mday != 1) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	from = tm->tm_year*1000 + tm->tm_yday+1;
}


/*
* lastweek: fills the from-to date values with dates ranging from the
*           first of last week (saturday) until the end (friday).
*/
lastweek()
{
	int t;
	struct tm *tm;
	struct tm *localtime();

	t = time() - DAYSECS;
	tm = localtime(t);

	while (tm->tm_wday != 5) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	to = tm->tm_year*1000 + tm->tm_yday+1;

	while (tm->tm_wday != 6) {
		t -= DAYSECS;
		tm = localtime(t);
	}
	from = tm->tm_year*1000 + tm->tm_yday+1;
}


/*
* maker: fills the from-to date values with dates ranging "n" days
*        ago until yesterday.
*/
maker(n)
{
	int i, t;
	struct tm *tm;
	struct tm *localtime();

	t = time() - DAYSECS;
	tm = localtime(t);
	to = tm->tm_year*1000 + tm->tm_yday+1;

	for (i=1; i<n; i++)
		t -= DAYSECS;
	tm = localtime(t);
	from = tm->tm_year*1000 + tm->tm_yday+1;
}


/*
* maked: loads the "fyr-fday-tyr-tday" variables with the range
*        given in the julian "from-to" dates.
*/
maked(fj, tj)
{
	char *greg();

	if (!sfrom[0]) strcpy(sfrom, greg(fj));
	if (!sto[0])   strcpy(sto  , greg(tj? tj:fj));

	fyr  = fj/1000;
	fday = fj%1000;
	if (to) {
		tyr  = tj/1000;
		tday = tj%1000;
	} else {
		tyr  = fyr;
		tday = fday;
	}
}


/*
* getdata: sums all the "cmds" data for the given range of days into
*          the "tots" structure.
*/
getdata()
{
	int n, i, max;

	while ((fyr*1000+fday) <= (tyr*1000+tday)) {
		if (getaday(fyr, fday)==0) {
			for (n=0; n<MAXCMDS; n++)
			if (cmds[n].name[0]) {
				if ((i = enter(cmds[n].name)) < 0)
					printf("Command Table Overflow.\n");
				else {
					tots[i].count += cmds[n].count;
					tots[i].realt += cmds[n].realt;
					tots[i].cput  += cmds[n].cput;
					tots[i].syst  += cmds[n].syst;
					tots[i].io    += cmds[n].io  ;
				}
			}
		}
		max = (fyr%4)==0 ? 366 : 365;
		if (++fday > max) {
			++fyr;
			fday = 1;
		}
	}
}


/*
* getaday: reads in one days worth of data
*/
getaday(yr, day)
{
	int  lotime, hitime, users, n;
	FILE *tfd;

	sprintf(dayfile, daymodel, yr*1000 + day);

	zeroout((int *)cmds, sizeof cmds);
	if ((tfd = fopen(dayfile, "r"))==NULL)
		return(1);

	lotime = getw(tfd);
	hitime = getw(tfd);
	users  = getw(tfd);
	if (lotime==EOF || hitime==EOF || users==EOF) {
		fclose(tfd);
		return(1);
	}
	fread((char *)aday, sizeof *aday, users, tfd);
	while((n = getw(tfd)) != EOF) {
		if (n>=0 && n<MAXCMDS) {
			fread((char *)&cmds[n], sizeof *cmds, 1, tfd);
			/*
			 *  Validate the record
			 */
			if (cmds[n].count < 1 || cmds[n].count > 40000) {
				bad++;
				zeroout((int *)&cmds[n], sizeof *cmds);
			}
		}
		else
			bad++;
	}
	fclose(tfd);
	return(0);
}


/*
* enter: finds a place for a command in the "tots" table.
*/
enter(cmd) char *cmd;
{
	int i, j, k;

	i = k = 0;
	for (j=0; j<8; j++)
		i = i*7 + cmd[j];
	if (i<0)
		i = -i;
	for (i %= MAXCMDS; tots[i].name[0]; i = (i+1)%MAXCMDS) {
		if (strcmp(cmd, tots[i].name)==0)
			goto yes;
		if (++k > MAXCMDS) return(-1);
	}
	strcpy(tots[i].name, cmd);

yes:    if ((i<0) || (i>MAXCMDS)) return(-1);
	return(i);
}


/*
* strip: removes unwanted commands used 'n' times or fewer.  If
*        'proflag' is not 0, the user will be prompted before it
*        is removed.
*/
strip(n, proflag)
{
	int  i, j;
	char ans[20];

	if ((j = enter("***other")) < 0) {
		printf("Command Table Overflow.\n");
		return;
	}
	for (i=0; i<MAXCMDS; i++)
		if (tots[i].name[0] && tots[i].count <= n) {
			if (proflag) {
				printf("%-8s -- ", tots[i].name);
				gets(ans);
				if (ans[0]!='y')
					continue;
			}
			tots[i].name[0] = NULL;
			tots[j].count += tots[i].count;
			tots[j].realt += tots[i].realt;
			tots[j].cput  += tots[i].cput ;
			tots[j].syst  += tots[i].syst ;
		}
}


/*
* sort: sorts an "index" array pointing to the "tots" elements.
*/
sort()
{
	int n;
	extern ncmp(), bcmp(), tcmp();

	for (n=0; n<MAXCMDS; n++)
		if (tots[n].name[0])
			index[numb++] = n;
	qsort(index, numb, sizeof *index, nflg? ncmp:(bflg? bcmp:tcmp));
}


/*
* ncmp: compares the "tots" elements based on the number of calls.
*/
ncmp(i1, i2) int *i1, *i2;
{
	int i, j;

	i = *i1;
	j = *i2;
	if(tots[i].count == tots[j].count)
		return(tcmp(i1, i2));
	return(xflg?
		      tots[j].count - tots[i].count :
		      tots[i].count - tots[j].count
		);
}


/*
* bcmp: compares the "tots" elements based on the sum of cpu times
*       divided by the number of calls.
*/
bcmp(i1, i2) int *i1, *i2;
{
	int i, j, n1, n2;

	i = *i1;
	j = *i2;

	n1 = (tots[i].cput + tots[i].syst) / tots[i].count;
	n2 = (tots[j].cput + tots[j].syst) / tots[j].count;

	if (n1 < n2)
		return(xflg? -1: 1);
	if (n1 > n2)
		return(xflg? 1: -1);
	return(0);
}


/*
* tcmp: compares the "tots" elements based on the sum of cpu times.
*/
tcmp(i1, i2) int *i1, *i2;
{
	int i, j, n1, n2;

	i = *i1;
	j = *i2;

	n1 = tots[i].cput + tots[i].syst;
	n2 = tots[j].cput + tots[j].syst;

	if (n1 < n2)
		return(xflg? -1: 1);
	if (n1 > n2)
		return(xflg? 1: -1);
	return(0);
}


/*
* totals: counts the totals for each category
*/
totals()
{
	int  n;
	struct cmds *tp;

	tcount = trealt = tcput = tsyst = tio = 0;
	for (n=0; n<numb; n++) {
		tp = &tots[index[n]];
		tcount += tp->count;
		trealt += tp->realt;
		tcput  += tp->cput;
		tsyst  += tp->syst;
		tio    += tp->io;
	}
}


/*
* output: writes out the data ... what else (he said knowingly)
*/
output()
{
	int n;
	struct cmds *tp;

	if (strcmp(sfrom, sto)==0)
		printf("Command Usage Summary for %s.\n\n", sfrom);
	else    printf("Command Usage Summary for %s to %s.\n\n", sfrom, sto);

        if (eflg)
                printf("Average cpu time and disk io per call and elapsed/cpu and io/cpu ratios");
        else {
                printf("Total cpu and elapsed time for each command");
                if (lflg)
                        printf(" with cpu time separated into user and system time");
                if (cflg)
                        printf(" and percentages of TOTALS");
                if (tflg)
                        printf(" and ratio of elapsed time to cpu time");
	}
        printf(".\n");

        if (nflg)
                printf("Sorted by number of calls");
        else
                if (bflg)
                        printf("Sorted by cpu time divided by the number of calls");
                else    printf("Sorted by cpu time");
        if (xflg)
                printf(" in reverse order.\n\n\n");
        else    printf(".\n\n\n");

	/*
	*  This is all getting so cluttered ... oh well ... since the "e"
	*  option was an afterthought, treat it as a special case.
	*/
	if (eflg) {
		evaluate();
		return;
	}

	printf("           #commands");
	if (cflg) printf("   %% of ");
	printf("       elapsed");
	if (cflg) printf("   %% of ");
	if (lflg) {
		printf("      user cpu");
		if (cflg) printf("   %% of ");
		printf("    system cpu");
	} else
		printf("      cpu time");
	if (cflg) printf("   %% of ");
	printf("   disk io");
	if (cflg) printf("   %% of ");
	if (tflg) printf("  elapsed/cpu");
	printf("\n");

	printf("%20s","");
	if (cflg) printf("   TOTAL");
	printf("   time (mins)");
	if (cflg) printf("   TOTAL");
	if (lflg) {
		printf("   time (secs)");
		if (cflg) printf("   TOTAL");
		printf("   time (secs)");
	} else
		printf("        (secs)");
	if (cflg) printf("   TOTAL");
	printf("%10s", "");
	if (cflg) printf("   TOTAL");
	if (tflg) printf("        ratio");
	printf("\n\n");

	printf("TOTALS        ");
	column(tcount, trealt, tcput, tsyst, tio);
	printf("\n\n\n");

	for (n=0; n<numb; n++) {
		tp = &tots[index[n]];
		printf("%-14s", tp->name);
		column(tp->count, tp->realt, tp->cput, tp->syst, tp->io);
	}
}


/*
* column: outputs the info in neat little columns
*/
column(n, a, b, c, d) int n, a, b, c, d;
{
	int  i;
	float percent();

	printf("%6d", n);
	if (cflg) {
		if (n == tcount)
			printf("%8s", ""); else
			printf("%7.02f%%", percent(n,tcount));
	}
	col(a, trealt, sixty);
	if (lflg) {
		col(b, tcput, athousand);
		col(c, tsyst, athousand);
	} else
		col(b+ c, tcput+ tsyst, athousand);
	printf("%10d", d);
	if (cflg) {
		if (d == tio)
			printf("%8s", ""); else
			printf("%7.02f%%", percent(d,tio));
	}
	if (tflg) {
		i = (float) a / ((b + c) / (float)1000) + .5;
		printf("%11d/1", i);
	}
	printf("\n");
}


col(a, m, scale) int a, m; float scale;
{
	printf("%14.02f", a / scale);
	if (cflg) {
		if (a == m)
			printf("%8s", ""); else
			printf("%7.02f%%", percent(a,m));
	}
}


/*
* evaluate: rates each command on an individual basis giving the
*           average cpu, average io, and the real/cpu and io/cpu
*           ratios.
*/
evaluate()
{
	int  n;
	struct cmds *tp;

	printf("           #commands   average cpu    average io   elapsed/cpu   disk io/cpu\n");
        printf("                     time per call      per call         ratio         ratio");
	printf("\n\n");
	printf("TOTALS        ");
	evacol(tcount, trealt, tcput, tsyst, tio);
	printf("\n\n\n");

	for (n=0; n<numb; n++) {
		tp = &tots[index[n]];
		printf("%-14s", tp->name);
		evacol(tp->count, tp->realt, tp->cput, tp->syst, tp->io);
	}
}


/*
* evacol: outputs the evaluation data in neat little columns.
*/
evacol(n, a, b, c, d) int n, a, b, c, d;
{
	int   i;
	float f;

	printf("%6d", n);
	f = ((b+c)/n)/athousand;
	printf("%14.02f", f);
	printf("%14.02f", (float)d/n);
	i = (float)a / ((b + c) / (float)1000) +.5;
	printf("%12d/1", i);
	i = (float)d  / ((b + c) / (float)100) +.5;
	printf("%12d/1", i);
	printf("\n");
}


/*
* zeroout: fills the buffer pointed to with zeros
*/
zeroout(s, len) int *s;
{
	register i;

	for (i=0; i<(len/BYTEPWD); i++)
		*s++ = 0;
}


int  kmr[13] = {
	0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365,
};
int  kml[13] = {
	0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366,
};


/*
* julian: converts the given gregorian date into julian and returns
*         the date as an integer.
*/
julian(gdate) char *gdate;
{
	int  gmonth, gday, gyear, days;
	char *s;

	s = gdate;
	gmonth = atoi(s) - 1;
	while (*s++ != '/') ;
	gday   = atoi(s);
	while (*s++ != '/') ;
	gyear  = atoi(s);

	days = (gyear%4)==0 ? kml[gmonth] : kmr[gmonth];

	return(gyear*1000 + days + gday);
}


/*
* greg: converts the given integer julian date to a gregorian
*       character date and returns a pointer to the string.
*/
char *greg(jdate)
{
	static char gdate[9];
	int  month, day, year, dayno, *kmp, i;

	year  = jdate/1000;
	dayno = jdate%1000;

	kmp = (year%4)==0 ? kml : kmr;

	for (i=0; i<12; i++) {
		if (dayno > kmp[i] && dayno <= kmp[i+1]) {
			month = i+1;
			day = dayno - kmp[i];
			break;
		}
	}
	sprintf(gdate, "%d/%d/%d", month, day, year);
	return(gdate);
}


/*
* any: scans the given string for the given character and returns 1 if
*      it is found, 0 otherwise.
*/
any(s, c)
char *s, c;
{

	while(*s)
		if(*s++ == c)
			return(1);
	return(0);
}


/*
* percent: calculate  i/j * 100.0
*/
float percent(i, j) int i, j;
{
	float f;

	if (i == 0  ||  j == 0)
		return(zero);
	f = ((float)i / (float)j) * 100;
	return(f);
}
